import { ONE_DAY, ONE_SECOND } from 'constants/constants';

import { CarTop } from 'entities/Ride/api/useRidesSummary/useRidesSummary';

import { getArrayTop } from 'shared/helpers/getArrayTop/getArrayTop';
import { sortCarTop } from 'shared/helpers/sortCarTop/sortCarTop';
import { useFetchRequest } from 'shared/hooks/useFetchRequest/useFetchRequest';
import { ChartPoint } from 'shared/types/ChartPoint';

interface FleetStatsByCar {
    name: string;
    rank: number;
    number: string;
    model_id: string;
    object_id: string;
}

interface FleetStatsMedianRank {
    rank: number;
    timestamp: number;
    day: number;
}

interface UseFleetStatsData {
    since?: Nullable<number>;
    until?: Nullable<number>;
}

interface UseFleetStatsReq {
    company_tag_names?: string;
    report_type?: number; // 1 - median_ranks, 2 - by_car, 4 - distribution, 7 - all stats
    since?: number;
    until?: number;
}

interface UseFleetStatsRes {
    topScoringCars: CarTop[];
    worseScoringCars: CarTop[];

    medianFleetAverageForPeriod: number;

    medianFleetChartPoints: ChartPoint[];
    distributionChartPoints: ChartPoint[];
    todayScore?: number;
}

interface UseFleetStatsServerRes {
    by_object?: FleetStatsByCar[];
    median_ranks?: FleetStatsMedianRank[];
    distribution?: number[]; // length 10 (0-9 ranks)
    today_score?: number;
}

const TOP_COUNT = 5;

function fleetStatsParser(res: UseFleetStatsServerRes): UseFleetStatsRes {
    const { by_object = [], median_ranks = [], distribution = [], today_score } = res;
    const scoringCars: CarTop[] = by_object.map((car) => {
        return {
            name: car.name,
            model_id: car.model_id,
            object_id: car.object_id,
            value: car.rank,
            number: car.number,
        };
    });

    sortCarTop(scoringCars);

    const totalRanks = median_ranks.reduce((memo, rank) => {
        return memo + rank.rank;
    }, 0);

    return {
        topScoringCars: getArrayTop(scoringCars, TOP_COUNT, true),
        worseScoringCars: getArrayTop(scoringCars, TOP_COUNT),
        medianFleetAverageForPeriod: totalRanks / median_ranks.length,
        medianFleetChartPoints: median_ranks.map((rank) => {
            const date = rank.timestamp * ONE_SECOND;

            return {
                line1: rank.rank,
                begin_date: date,
                end_date: date + ONE_DAY - 1,
            };
        }),
        distributionChartPoints: distribution.map((dist, index) => {
            return {
                line1: dist,
                begin_date: index,
                end_date: index,
            };
        }),
        todayScore: today_score,
    };
}

export function useFleetStats(data: UseFleetStatsData) {
    const req: UseFleetStatsReq = {};

    if (data.since && data.until) {
        req.since = data.since;
        req.until = data.until;
    }

    return useFetchRequest<UseFleetStatsReq, UseFleetStatsRes, UseFleetStatsServerRes>(
        '/api/leasing/fleet/stats',
        req,
        {
            parser: fleetStatsParser,
        },
    );
}
