import * as React from 'react';
import cn from 'classnames';

import { useFleetStats } from 'features/DashboardFleetScoring/api/useFleetStats/useFleetStats';

import { formatCarNumber } from 'entities/Car';
import { CarTop } from 'entities/Ride/api/useRidesSummary/useRidesSummary';

import { DateAggregation } from 'shared/consts/DateAggregation';
import { IslandTheme } from 'shared/consts/IslandTheme';
import { Path } from 'shared/consts/Path';
import { ScoringSize } from 'shared/consts/ScoringSize';
import { generateRouterPath } from 'shared/helpers/generateRouterPath/generateRouterPath';
import { getChartAggregatedPoints } from 'shared/helpers/getChartAggregatedPoints/getChartAggregatedPoints';
import { DateAggregationConfig } from 'shared/helpers/getDateAggregation/getDateAggregation';
import { useRetry } from 'shared/hooks/useRetry/useRetry';
import { ChartKey } from 'shared/types/ChartPoint';
import { ChartViewHistogram, ChartViewType } from 'shared/ui/ChartViewHistogram/ChartViewHistogram';
import { Island } from 'shared/ui/Island/Island';
import { Link } from 'shared/ui/Link/Link';
import { Scoring } from 'shared/ui/Scoring/Scoring';
import { Spin } from 'shared/ui/Spin/Spin';
import { TopTable, TopTableItem } from 'shared/ui/TopTable/TopTable';

import { i18n } from 'features/DashboardFleetScoring/ui/DashboardFleetScoring/DashboardFleetScoring.i18n';

import styles from 'features/DashboardFleetScoring/ui/DashboardFleetScoring/DashboardFleetScoring.css';

export interface DashboardFleetScoringProps {
    className?: string;

    aggregationConfig: DateAggregationConfig;
}

const CHART_KEYS: ChartKey[] = ['line1'];
const HIGH_RISK_LINE = 3;
const BAR_COLORS = [
    '#fc3f1d',
    '#FC471B',
    '#FD6915',
    '#FE860E',
    '#FEA109',
    '#FFBE03',
    '#ECCA06',
    '#C2C514',
    '#97BF22',
    '#6CBA30',
    '#45B53D',
];

function getTopTableItems(top: Optional<CarTop[]>): TopTableItem[] {
    if (!top) {
        return [];
    }

    return top.map((car, index): TopTableItem => {
        return {
            id: car.object_id,
            text: (
                <Link
                    href={generateRouterPath(Path.CAR, { id: car.object_id })}
                    className={styles.car}
                >
                    <div className={styles.carName}>{car.name}</div>
                    <div className={styles.carNumber}>{formatCarNumber(car.number)}</div>
                </Link>
            ),
            value: <Scoring value={car.value} />,
        };
    });
}

export const DashboardFleetScoring: React.FC<DashboardFleetScoringProps> = function DashboardFleetScoring({
    className,
    aggregationConfig,
}) {
    const { data, isLoading, error, reload } = useFleetStats({
        since: aggregationConfig.since,
        until: aggregationConfig.until,
    });
    useRetry(error, reload);

    const {
        todayScore,
        topScoringCars,
        worseScoringCars,
        medianFleetAverageForPeriod,
        distributionChartPoints = [],
        medianFleetChartPoints = [],
    } = data || {};

    const medianPoints = React.useMemo(
        () =>
            aggregationConfig.aggregation === DateAggregation.DAY
                ? medianFleetChartPoints
                : getChartAggregatedPoints(medianFleetChartPoints, aggregationConfig),
        [aggregationConfig, medianFleetChartPoints],
    );

    return (
        <div className={cn(styles.container, className)}>
            <div className={styles.column}>
                <div className={styles.row}>
                    <TopTable
                        className={styles.top}
                        title={i18n('Top cars by scoring')}
                        items={getTopTableItems(topScoringCars)}
                        isLoading={isLoading}
                    />
                    <TopTable
                        className={styles.top}
                        title={i18n('Worst cars by scoring')}
                        items={getTopTableItems(worseScoringCars)}
                        isLoading={isLoading}
                    />
                </div>

                <Island title={i18n('Driver’s distribution by scoring')}>
                    {!isLoading ? (
                        <ChartViewHistogram
                            className={styles.chartDistribution}
                            points={distributionChartPoints}
                            barColors={BAR_COLORS}
                            keys={CHART_KEYS}
                            viewType={ChartViewType.VALUE}
                            aggregationConfig={aggregationConfig}
                            keysFormat={[
                                {
                                    fraction: 0,
                                },
                            ]}
                        />
                    ) : (
                        <div className={styles.chartShimmer}>
                            <Spin />
                        </div>
                    )}
                </Island>
            </div>

            <div className={styles.column}>
                <Island
                    title={
                        <div className={styles.titleContainer}>
                            <h3 className={styles.title}>{i18n('Median fleet scoring')}</h3>

                            <div className={styles.risk}>{i18n('High risk of accidents')}</div>
                        </div>
                    }
                >
                    <div className={styles.tabletRow}>
                        <Island
                            className={styles.tablet}
                            title={
                                <h3 className={styles.islandSubTitle}>
                                    {isLoading ? (
                                        <Spin />
                                    ) : (
                                        <Scoring
                                            value={medianFleetAverageForPeriod}
                                            size={ScoringSize.L}
                                        />
                                    )}
                                </h3>
                            }
                            theme={IslandTheme.GRAY}
                        >
                            {i18n('Avg. per day')}
                        </Island>

                        <Island
                            className={styles.tablet}
                            title={
                                <h3 className={styles.islandSubTitle}>
                                    {isLoading ? (
                                        <Spin />
                                    ) : (
                                        <Scoring
                                            value={todayScore}
                                            size={ScoringSize.L}
                                        />
                                    )}
                                </h3>
                            }
                            theme={IslandTheme.GRAY}
                        >
                            {i18n('Today')}
                        </Island>
                    </div>

                    {!isLoading ? (
                        <ChartViewHistogram
                            className={styles.chartMedian}
                            points={medianPoints}
                            aggregationConfig={aggregationConfig}
                            dottedLine={HIGH_RISK_LINE}
                            keys={CHART_KEYS}
                            keysFormat={[
                                {
                                    maxValue: 10,
                                    fraction: 1,
                                },
                            ]}
                        />
                    ) : (
                        <div className={styles.chartShimmer}>
                            <Spin />
                        </div>
                    )}
                </Island>
            </div>
        </div>
    );
};
