import * as React from 'react';
import { Placement } from '@floating-ui/react-dom';

import { ONE_SECOND } from 'constants/constants';

import { DateFilter } from 'shared/consts/DateFilter';
import { InputSize } from 'shared/consts/InputSize';
import { getDateEndDay } from 'shared/helpers/getDateEndDay/getDateEndDay';
import { getDateStartDay } from 'shared/helpers/getDateStartDay/getDateStartDay';
import { FilterGroup, FilterGroupProps } from 'shared/ui/FilterGroup/FilterGroup';
import { InputDateRange } from 'shared/ui/InputDateRange/InputDateRange';

import { i18n } from 'features/FilterDate/ui/FilterDate/FilterDate.i18n';

import styles from 'features/FilterDate/ui/FilterDate/FilterDate.css';

export interface FilterDateOptions {
    [DateFilter.SINCE]: Nullable<number>;
    [DateFilter.UNTIL]: Nullable<number>;
}

export interface FilterDateProps extends Pick<FilterGroupProps, 'title'> {
    className?: string;
    type: string;
    filters?: FilterDateOptions;
    placement?: Placement;

    inline?: boolean;

    onChangeFilter(type: string, value: Record<DateFilter, Nullable<number>>): void;
}

export const FilterDate: React.FC<FilterDateProps> = function FilterDate({
    className,
    title,
    filters,
    type,
    inline,
    placement,
    onChangeFilter,
}) {
    const onDateChangeHandler = React.useCallback(
        (values: [Optional<Date>, Optional<Date>]) => {
            onChangeFilter(type, {
                [DateFilter.SINCE]: values[0] ? getDateStartDay(values[0]).getTime() / ONE_SECOND : null,
                [DateFilter.UNTIL]: values[1] ? Math.trunc(getDateEndDay(values[1]).getTime() / ONE_SECOND) : null,
            });
        },
        [onChangeFilter],
    );

    const placeholders = React.useMemo(() => [i18n('From'), i18n('To')], []);

    const dates = React.useMemo(() => {
        const sinceDate = filters?.since ? new Date(filters.since * ONE_SECOND) : null;
        const untilDate = filters?.until ? new Date(filters.until * ONE_SECOND) : null;

        return [sinceDate, untilDate];
    }, [filters]);

    return (
        <FilterGroup
            className={className}
            title={title}
        >
            <InputDateRange
                className={inline ? styles.filterInlineDate : styles.filterDate}
                inputSize={InputSize.M}
                placeholders={placeholders}
                dates={dates}
                placement={placement}
                onDateChange={onDateChangeHandler}
            />
        </FilterGroup>
    );
};
