import * as React from 'react';
import cn from 'classnames';

import { postNotifier } from 'features/NotificationsTelegram/api/postNotifier/postNotifier';
import { updateNotifier } from 'features/NotificationsTelegram/api/updateNotifier/updateNotifier';
import { updateNotifierBotId } from 'features/NotificationsTelegram/api/updateNotifierBotId/updateNotifierBotId';
import { useNotifiersList } from 'features/NotificationsTelegram/api/useNotifiersList/useNotifiersList';
import { NOTIFIER_TELEGRAM_TYPE } from 'features/NotificationsTelegram/consts/constants';
import { NotifierSchema } from 'features/NotificationsTelegram/types/NotifierSchema';
import { NotificationsTelegramForm } from 'features/NotificationsTelegram/ui/NotificationsTelegramForm/NotificationsTelegramForm';
import { NotificationsTelegramModal } from 'features/NotificationsTelegram/ui/NotificationsTelegramModal/NotificationsTelegramModal';

import { getFetchErrorMessage } from 'shared/helpers/getFetchErrorMessage/getFetchErrorMessage';
import { useFormController } from 'shared/hooks/useFormController/useFormController';
import { ErrorMessage } from 'shared/ui/ErrorMessage/ErrorMessage';
import { Spin } from 'shared/ui/Spin/Spin';

import styles from 'features/NotificationsTelegram/ui/NotificationsTelegram/NotificationsTelegram.css';

export interface NotificationsTelegramProps {
    className?: string;
}

export interface NotificationsTelegramFormSchema {
    groupName: string;
    groupId: string;

    // meta type for messages based on server errors
    _serverError?: never;
}

export interface NotificationsTelegramBotIdFormSchema {
    botId: string;

    // meta type for messages based on server errors
    _serverError?: never;
}

export const NotificationsTelegram: React.FC<NotificationsTelegramProps> = function NotificationsTelegram({
    className,
}) {
    const [currentGroup, setCurrentGroup] = React.useState<NotifierSchema>();
    const [isModalOpen, setModalOpen] = React.useState<boolean>(false);
    const [isUiLoading, setUiLoading] = React.useState<boolean>(false);

    const { data, isLoading, reload, error } = useNotifiersList({ type: NOTIFIER_TELEGRAM_TYPE });
    const { objects = [] } = data || {};

    const { getValues, validate, setError, controller } =
        useFormController<OptionalRecord<NotificationsTelegramBotIdFormSchema>>();

    const initForm = React.useMemo(
        (): NotificationsTelegramBotIdFormSchema => ({
            botId: objects.length > 0 ? objects[0].meta.bot.bot_id : '',
        }),

        [objects],
    );

    const onAddGroup = React.useCallback(() => {
        setModalOpen(true);
    }, []);

    const onModalClose = React.useCallback(() => {
        setCurrentGroup(undefined);
        setModalOpen(false);
    }, []);

    const onDelete = React.useCallback(() => {
        reload();
    }, [reload]);

    const onLinkGroupHandler = React.useCallback(
        async (formData: NotificationsTelegramFormSchema) => {
            if (validate()) {
                const formDataBot = getValues();

                await postNotifier({
                    botId: formDataBot?.botId!,
                    ...formData,
                });

                reload();
            }
        },
        [getValues, reload, validate],
    );

    const onUpdateGroupHandler = React.useCallback(
        async (notifier: NotifierSchema) => {
            if (validate()) {
                const formDataBot = getValues();

                await updateNotifier({
                    ...notifier,
                    meta: {
                        ...notifier.meta,
                        bot: {
                            bot_id: formDataBot?.botId!,
                        },
                    },
                });

                reload();
            }
        },
        [getValues, reload, validate],
    );

    const onUpdateBotId = React.useCallback(async () => {
        if (validate()) {
            const formDataBot = getValues();
            const newId = formDataBot?.botId!;

            if (initForm.botId === newId) {
                return;
            }

            setUiLoading(true);

            try {
                await updateNotifierBotId({
                    bot_id: initForm.botId,
                    new_bot_id: newId,
                });

                reload();
            } catch (error) {
                setError('_serverError', getFetchErrorMessage(error));
            }

            setUiLoading(false);
        }
    }, [getValues, initForm.botId, reload, setError, validate]);

    const onEditGroupClick = React.useCallback((group: NotifierSchema) => {
        setCurrentGroup(group);
        setModalOpen(true);
    }, []);

    if (error) {
        return (
            <div className={cn(styles.container, className)}>
                <ErrorMessage error={getFetchErrorMessage(error)} />
            </div>
        );
    }

    return (
        <div className={cn(styles.container, className)}>
            {!data ? (
                <div className={styles.formSpinner}>
                    <Spin />
                </div>
            ) : (
                <NotificationsTelegramForm
                    isLoading={isLoading || isUiLoading}
                    groups={objects}
                    initial={initForm}
                    controller={controller}
                    onAddGroup={onAddGroup}
                    onUpdateBotId={onUpdateBotId}
                    onEditGroupClick={onEditGroupClick}
                />
            )}

            {isModalOpen && (
                <NotificationsTelegramModal
                    group={currentGroup}
                    onClose={onModalClose}
                    onLink={onLinkGroupHandler}
                    onUpdate={onUpdateGroupHandler}
                    onDelete={onDelete}
                />
            )}
        </div>
    );
};
