import * as React from 'react';

import { NotifierSchema } from 'features/NotificationsTelegram/types/NotifierSchema';
import { NotificationsTelegramBotIdFormSchema } from 'features/NotificationsTelegram/ui/NotificationsTelegram/NotificationsTelegram';

import { ButtonColor } from 'shared/consts/ButtonColor';
import { ButtonSize } from 'shared/consts/ButtonSize';
import { InputSize } from 'shared/consts/InputSize';
import { validateRequired } from 'shared/helpers/validateRequired/validateRequired';
import { useForm, UseFormValidation } from 'shared/hooks/useForm/useForm';
import { UseFormControllerSub } from 'shared/hooks/useFormController/useFormController';
import { Button } from 'shared/ui/Button/Button';
import { ErrorLabel } from 'shared/ui/ErrorLabel/ErrorLabel';
import { ErrorMessage } from 'shared/ui/ErrorMessage/ErrorMessage';
import { FormContainer } from 'shared/ui/FormContainer/FormContainer';
import { FormErrorContainer } from 'shared/ui/FormErrorContainer/FormErrorContainer';
import { FormLabel } from 'shared/ui/FormLabel/FormLabel';
import { Input } from 'shared/ui/Input/Input';
import { Spin } from 'shared/ui/Spin/Spin';
import { Table } from 'shared/ui/Table/Table';

import { i18n } from 'features/NotificationsTelegram/ui/NotificationsTelegramForm/NotificationsTelegramForm.i18n';

import EditIcon from 'shared/ui/Icons/images/edit-outline-16.inline.svg';
import PlusIcon from 'shared/ui/Icons/images/plus-outline-16.inline.svg';

import styles from 'features/NotificationsTelegram/ui/NotificationsTelegramForm/NotificationsTelegramForm.css';

export interface NotificationsTelegramFormProps {
    className?: string;
    isLoading?: boolean;
    groups: NotifierSchema[];
    initial?: NotificationsTelegramBotIdFormSchema;
    controller?: UseFormControllerSub<NotificationsTelegramBotIdFormSchema>;

    onAddGroup?(): void;
    onUpdateBotId?(): void;

    onEditGroupClick(group: NotifierSchema): void;
}

const INIT_FORM: OptionalRecord<NotificationsTelegramBotIdFormSchema> = {
    botId: '',
};

const VALIDATION_RULES: UseFormValidation<OptionalRecord<NotificationsTelegramBotIdFormSchema>> = {
    botId: validateRequired(),
};

interface GroupRowProps {
    group: NotifierSchema;
    onClick(group: NotifierSchema): void;
}

const GroupRow: React.FC<GroupRowProps> = function GroupRow({ group, onClick }: GroupRowProps) {
    const onEditGroup = React.useCallback(() => {
        onClick(group);
    }, [group, onClick]);

    return (
        <Table.Row
            className={styles.row}
            onClick={onEditGroup}
        >
            <Table.Td gap />
            <Table.Td>{group.display_name}</Table.Td>
            <Table.Td>{group.meta.chat.chat_id}</Table.Td>
            <Table.Td className={styles.action}>
                <Button
                    className={styles.edit}
                    color={ButtonColor.CLEAR}
                    size={ButtonSize.M}
                    icon={EditIcon}
                />
            </Table.Td>
            <Table.Td gap />
        </Table.Row>
    );
};

export const NotificationsTelegramForm: React.FC<NotificationsTelegramFormProps> = function NotificationsTelegramForm({
    className,
    isLoading,
    groups,
    initial,
    controller,
    onAddGroup,
    onUpdateBotId,
    onEditGroupClick,
}) {
    const hasGroups = groups.length > 0;
    const { update, watch, errors } = useForm<OptionalRecord<NotificationsTelegramBotIdFormSchema>>({
        init: { ...INIT_FORM, ...initial },
        validation: VALIDATION_RULES,
        controller,
    });

    return (
        <FormContainer className={className}>
            <FormLabel title={i18n('Telegram bot ID')}>
                <div className={styles.item}>
                    <FormErrorContainer error={<ErrorLabel errors={errors.botId} />}>
                        <Input
                            inputSize={InputSize.M}
                            placeholder={i18n('bot ID')}
                            value={watch('botId')}
                            hasError={Boolean(errors.botId)}
                            disabled={isLoading}
                            hasClear
                            onInputChange={update('botId')}
                        />
                    </FormErrorContainer>
                </div>

                <div className={styles.description}>
                    {i18n(
                        'To set up Telegram notifications you have to add a Telegram bot. You can create the bot directly in the Telegram App. The token you need can be found on the setting page of your bot.',
                    )}
                </div>
            </FormLabel>

            <FormLabel
                title={
                    <div className={styles.groups}>
                        <div>{i18n('Linked groups')}</div>
                        {hasGroups && (
                            <Button
                                className={styles.groupsAdd}
                                color={ButtonColor.PRIMARY}
                                size={ButtonSize.M}
                                icon={PlusIcon}
                                label={i18n('Link a group')}
                                onClick={onAddGroup}
                            />
                        )}
                    </div>
                }
            >
                {isLoading ? (
                    <div className={styles.loading}>
                        <Spin />
                    </div>
                ) : hasGroups ? (
                    <Table className={styles.table}>
                        <Table.Head>
                            <Table.Row>
                                <Table.Th gap />
                                <Table.Th>{i18n('Group name')}</Table.Th>
                                <Table.Th>{i18n('ID')}</Table.Th>
                                <Table.Th></Table.Th>
                                <Table.Th gap />
                            </Table.Row>
                        </Table.Head>
                        <Table.Body>
                            {groups.map((group) => {
                                return (
                                    <GroupRow
                                        key={group.name}
                                        group={group}
                                        onClick={onEditGroupClick}
                                    />
                                );
                            })}
                        </Table.Body>
                    </Table>
                ) : (
                    <div className={styles.item}>
                        <Button
                            className={styles.linkFirstGroup}
                            color={ButtonColor.SECONDARY}
                            size={ButtonSize.M}
                            disabled={!Boolean(watch('botId'))}
                            label={i18n('Link the first group')}
                            wide
                            onClick={onAddGroup}
                        />
                    </div>
                )}
            </FormLabel>

            <Button
                color={ButtonColor.PRIMARY}
                size={ButtonSize.M}
                label={i18n('Save changes')}
                disabled={!initial?.botId}
                wide
                onClick={onUpdateBotId}
            />

            <ErrorMessage error={errors._serverError} />
        </FormContainer>
    );
};
