import * as React from 'react';

import { deleteNotifier } from 'features/NotificationsTelegram/api/deleteNotifier/deleteNotifier';
import { getNotifierErrorMessage } from 'features/NotificationsTelegram/helpers/getNotifierErrorMessage/getNotifierErrorMessage';
import { NotifierSchema } from 'features/NotificationsTelegram/types/NotifierSchema';
import { NotificationsTelegramFormSchema } from 'features/NotificationsTelegram/ui/NotificationsTelegram/NotificationsTelegram';
import { NotificationsTelegramDeleteDialog } from 'features/NotificationsTelegram/ui/NotificationsTelegramDeleteDialog/NotificationsTelegramDeleteDialog';
import { TelegramGroupForm } from 'features/NotificationsTelegram/ui/TelegramGroupForm/TelegramGroupForm';

import { ButtonColor } from 'shared/consts/ButtonColor';
import { ButtonSize } from 'shared/consts/ButtonSize';
import { useEqualDataForm } from 'shared/hooks/useEqualDataForm/useEqualDataForm';
import { useFormController } from 'shared/hooks/useFormController/useFormController';
import { Button } from 'shared/ui/Button/Button';
import { ModalContainer, ModalContainerProps } from 'shared/ui/ModalContainer/ModalContainer';

import { i18n } from 'features/NotificationsTelegram/ui/NotificationsTelegramModal/NotificationsTelegramModal.i18n';

import TrashIcon from 'shared/ui/Icons/images/trash-outline-16.inline.svg';

import styles from 'features/NotificationsTelegram/ui/NotificationsTelegramModal/NotificationsTelegramModal.css';

export interface NotificationsTelegramModalProps extends Pick<ModalContainerProps, 'onClose'> {
    group?: NotifierSchema;

    onLink(data: NotificationsTelegramFormSchema): void;
    onUpdate(data: NotifierSchema): void;
    onDelete(): void;
}

export const NotificationsTelegramModal: React.FC<NotificationsTelegramModalProps> =
    function NotificationsTelegramModal({ onClose, group, onLink, onUpdate, onDelete }) {
        const isExist = Boolean(group);
        const [isDeleteDialogOpen, setDeleteDialogOpen] = React.useState<boolean>(false);
        const [isUiLoading, setUiLoading] = React.useState<boolean>(false);

        const { getValues, validate, setError, controller } =
            useFormController<OptionalRecord<NotificationsTelegramFormSchema>>();

        const initForm = React.useMemo(
            (): NotificationsTelegramFormSchema => ({
                groupName: group ? group.display_name : '',
                groupId: group ? group.meta.chat.chat_id : '',
            }),

            [group],
        );

        const onDeleteSubmitHandler = React.useCallback(async () => {
            if (!group) {
                return;
            }

            try {
                await deleteNotifier(group.name);
                // @todo: add notification

                if (onDelete) {
                    onDelete();
                }

                if (onClose) {
                    onClose();
                }
            } catch (error) {
                // return setError('_serverError', getFetchErrorMessage(error));
            }
        }, [group, onDelete, onClose]);

        const onDeleteDialogOpen = React.useCallback(() => {
            setDeleteDialogOpen(true);
        }, []);

        const onDeleteDialogClose = React.useCallback(() => {
            setDeleteDialogOpen(false);
        }, []);

        const onLinkHandler = React.useCallback(async () => {
            if (validate()) {
                const formData = getValues();

                setUiLoading(true);

                try {
                    await onLink(formData as NotificationsTelegramFormSchema);

                    if (onClose) {
                        onClose();
                    }
                } catch (error) {
                    setError('_serverError', getNotifierErrorMessage(error));
                }

                setUiLoading(false);
            }
        }, [getValues, onClose, onLink, setError, validate]);

        const onUpdateHandler = React.useCallback(async () => {
            if (group && validate()) {
                const formData = getValues();
                const { groupName, groupId } = formData as NotificationsTelegramFormSchema;

                setUiLoading(true);

                try {
                    await onUpdate({
                        ...group,
                        display_name: groupName,
                        meta: {
                            ...group.meta,
                            chat: {
                                ...group.meta.chat,
                                chat_id: groupId,
                            },
                        },
                    });

                    if (onClose) {
                        onClose();
                    }
                } catch (error) {
                    setError('_serverError', getNotifierErrorMessage(error));
                }

                setUiLoading(false);
            }
        }, [getValues, group, onClose, onUpdate, setError, validate]);

        const { isEqualData, onFormChangeHandler } = useEqualDataForm<NotificationsTelegramFormSchema>(initForm);

        return (
            <>
                <ModalContainer
                    className={styles.modal}
                    title={isExist ? i18n('Edit group') : i18n('Link group')}
                    hasClose
                    controls={
                        isExist ? (
                            <>
                                <Button
                                    color={ButtonColor.PRIMARY}
                                    size={ButtonSize.M}
                                    label={i18n('Save changes')}
                                    disabled={isEqualData || isUiLoading}
                                    onClick={onUpdateHandler}
                                />

                                <Button
                                    color={ButtonColor.ALARM}
                                    size={ButtonSize.M}
                                    icon={TrashIcon}
                                    label={i18n('Delete')}
                                    disabled={isUiLoading}
                                    onClick={onDeleteDialogOpen}
                                />
                            </>
                        ) : (
                            <>
                                <Button
                                    color={ButtonColor.PRIMARY}
                                    size={ButtonSize.M}
                                    label={i18n('Link a group')}
                                    disabled={isUiLoading}
                                    onClick={onLinkHandler}
                                />

                                <Button
                                    color={ButtonColor.SECONDARY}
                                    size={ButtonSize.M}
                                    label={i18n('Back')}
                                    disabled={isUiLoading}
                                    onClick={onClose}
                                />
                            </>
                        )
                    }
                    onClose={onClose}
                >
                    <TelegramGroupForm
                        isLoading={isUiLoading}
                        initial={initForm}
                        controller={controller}
                        onFormChange={onFormChangeHandler}
                    />
                </ModalContainer>

                {isDeleteDialogOpen && (
                    <NotificationsTelegramDeleteDialog
                        onDialogSubmit={onDeleteSubmitHandler}
                        onClose={onDeleteDialogClose}
                    />
                )}
            </>
        );
    };
