import * as React from 'react';

import { NotificationsTelegramFormSchema } from 'features/NotificationsTelegram/ui/NotificationsTelegram/NotificationsTelegram';

import { InputSize } from 'shared/consts/InputSize';
import { validateRequired } from 'shared/helpers/validateRequired/validateRequired';
import { useForm, UseFormValidation } from 'shared/hooks/useForm/useForm';
import { UseFormControllerSub } from 'shared/hooks/useFormController/useFormController';
import { ErrorLabel } from 'shared/ui/ErrorLabel/ErrorLabel';
import { ErrorMessage } from 'shared/ui/ErrorMessage/ErrorMessage';
import { FormContainer } from 'shared/ui/FormContainer/FormContainer';
import { FormErrorContainer } from 'shared/ui/FormErrorContainer/FormErrorContainer';
import { FormLabel } from 'shared/ui/FormLabel/FormLabel';
import { Input } from 'shared/ui/Input/Input';

import { i18n } from 'features/NotificationsTelegram/ui/TelegramGroupForm/TelegramGroupForm.i18n';

import styles from 'features/NotificationsTelegram/ui/TelegramGroupForm/TelegramGroupForm.css';

export interface TelegramGroupFormProps {
    className?: string;
    isLoading?: boolean;
    initial?: NotificationsTelegramFormSchema;
    controller?: UseFormControllerSub<NotificationsTelegramFormSchema>;

    onFormChange?(state: OptionalRecord<NotificationsTelegramFormSchema>): void;
}

const INIT_FORM: OptionalRecord<NotificationsTelegramFormSchema> = {
    groupName: '',
    groupId: '',
};

const VALIDATION_RULES: UseFormValidation<OptionalRecord<NotificationsTelegramFormSchema>> = {
    groupName: validateRequired(),
    groupId: validateRequired(),
};

export const TelegramGroupForm: React.FC<TelegramGroupFormProps> = function TelegramGroupForm({
    className,
    isLoading,
    initial,
    controller,
    onFormChange,
}) {
    const { update, watch, errors } = useForm<OptionalRecord<NotificationsTelegramFormSchema>>({
        init: { ...INIT_FORM, ...initial },
        validation: VALIDATION_RULES,
        controller,
        onFormChange,
    });

    return (
        <FormContainer className={className}>
            <FormLabel title={i18n('Group name')}>
                <div className={styles.item}>
                    <FormErrorContainer error={<ErrorLabel errors={errors.groupName} />}>
                        <Input
                            inputSize={InputSize.M}
                            value={watch('groupName')}
                            hasError={Boolean(errors.groupName)}
                            hasClear
                            onInputChange={update('groupName')}
                            disabled={isLoading}
                        />
                    </FormErrorContainer>
                </div>
            </FormLabel>

            <FormLabel title={i18n('Group ID')}>
                <div className={styles.item}>
                    <FormErrorContainer error={<ErrorLabel errors={errors.groupId} />}>
                        <Input
                            inputSize={InputSize.M}
                            value={watch('groupId')}
                            hasError={Boolean(errors.groupId)}
                            disabled={Boolean(initial?.groupId) || isLoading}
                            hasClear
                            onInputChange={update('groupId')}
                        />
                    </FormErrorContainer>
                </div>

                <div className={styles.description}>{i18n('To get a group ID long-tap on its title.')}</div>
            </FormLabel>

            <ErrorMessage error={errors._serverError} />
        </FormContainer>
    );
};
