import * as React from 'react';
import cn from 'classnames/bind';

import { fetchCarRentalAgreement } from 'entities/Car/api/fetchCarRentalAgreement/fetchCarRentalAgreement';

import { ButtonColor } from 'shared/consts/ButtonColor';
import { ButtonSize } from 'shared/consts/ButtonSize';
import { downloadFile } from 'shared/helpers/downloadFile/downloadFile';
import { getFetchErrorMessage } from 'shared/helpers/getFetchErrorMessage/getFetchErrorMessage';
import { useCacheRequestContext } from 'shared/hooks/useCacheRequestContext/useCacheRequestContext';
import { FormErrorSchema } from 'shared/types/FormErrorSchema';
import { Breadcrumbs } from 'shared/ui/Breadcrumbs/Breadcrumbs';
import { Button } from 'shared/ui/Button/Button';
import { ErrorMessage } from 'shared/ui/ErrorMessage/ErrorMessage';
import { ModalContainer, ModalContainerProps } from 'shared/ui/ModalContainer/ModalContainer';

import { i18n } from 'features/RentalRetiree/ui/RentalRetireeAgreementModal/RentalRetireeAgreementModal.i18n';

import styles from 'features/RentalRetiree/ui/RentalRetireeAgreementModal/RentalRetireeAgreementModal.css';

export interface RentalRetireeAgreementModalProps extends ModalContainerProps {
    offerId: string;
    userId: string;

    onFormSubmit(): void;
}

const cx = cn.bind(styles);

export const RentalRetireeAgreementModal: React.FC<RentalRetireeAgreementModalProps> =
    function RentalRetireeAgreementModal({ offerId, userId, onClose, onFormSubmit }) {
        const cache = useCacheRequestContext();

        const [isDownloaded, setDownloaded] = React.useState<boolean>(false);
        const [errorMessage, setErrorMessage] = React.useState<Optional<FormErrorSchema>>();

        const onDownloadHandler = React.useCallback(async () => {
            try {
                const file = await fetchCarRentalAgreement(offerId, userId, cache);

                downloadFile(file);

                setDownloaded(true);
                setErrorMessage(undefined);

                // @todo: add notification
            } catch (error) {
                return setErrorMessage(getFetchErrorMessage(error));
            }
        }, [offerId, userId, cache]);

        return (
            <ModalContainer
                className={styles.modal}
                title={
                    <>
                        <Breadcrumbs
                            className={styles.breadcrumbs}
                            items={[i18n('Starting the ride manually')]}
                        />

                        <h3 className={styles.title}>
                            {i18n('2 of 4. Download, print and sign the rental agreement')}
                        </h3>
                    </>
                }
                hasClose
                onClose={onClose}
            >
                <div className={styles.wrap}>
                    <div className={styles.document}>
                        <h5 className={styles.documentTitle}>{i18n('Rental agreement')}</h5>

                        <Button
                            className={cx(styles.download, { downloaded: isDownloaded })}
                            color={!isDownloaded ? ButtonColor.PRIMARY : undefined}
                            size={ButtonSize.M}
                            label={isDownloaded ? i18n('Downloaded') : i18n('Download')}
                            onClick={!isDownloaded ? onDownloadHandler : undefined}
                        />
                    </div>

                    <div className={styles.controls}>
                        <ErrorMessage error={errorMessage} />

                        <Button
                            color={isDownloaded ? ButtonColor.PRIMARY : ButtonColor.SECONDARY}
                            size={ButtonSize.M}
                            label={i18n('Confirm agreement was signed')}
                            wide
                            onClick={onFormSubmit}
                        />
                    </div>
                </div>
            </ModalContainer>
        );
    };
