import * as React from 'react';
import cn from 'classnames';

import { DocumentPhotoDataRes, useUserDocumentPhoto } from 'entities/User';
import { UserPhotoContentTypeSchema } from 'entities/User/types/UserPhotosSchema';

import { ButtonFile } from 'shared/ui/ButtonFile/ButtonFile';
import { ImagesPreviewUpload } from 'shared/ui/ImagesPreviewUpload/ImagesPreviewUpload';
import { PictureOptions } from 'shared/ui/Picture/Picture';

import { i18n } from 'features/RentalRetiree/ui/RentalRetireeDocumentPreviewItem/RentalRetireeDocumentPreviewItem.i18n';

import styles from 'features/RentalRetiree/ui/RentalRetireeDocumentPreviewItem/RentalRetireeDocumentPreviewItem.css';

export interface RentalRetireeDocumentPreviewItemProps extends DocumentPhotoDataRes {
    className?: string;
    file?: File;

    onChange?(file: File, type: UserPhotoContentTypeSchema): void;
}

const PHOTO_CAPTION: Record<UserPhotoContentTypeSchema, string> = {
    lb: i18n('Back'),
    lf: i18n('Front'),
    pb: i18n('Page with photo'),
    ps: i18n('Selfie'),
};

export const RentalRetireeDocumentPreviewItem: React.FC<RentalRetireeDocumentPreviewItemProps> =
    function RentalRetireeDocumentPreviewItem({ className, type, userId, photoId, file, onChange }) {
        const { data, isLoading } = useUserDocumentPhoto(userId, photoId);

        const [currentFiles, setCurrentFiles] = React.useState<File[]>([]);

        const label = PHOTO_CAPTION[type];

        React.useEffect(() => {
            if (file !== currentFiles[0]) {
                setCurrentFiles(file ? [file] : []);
            }
        }, [file]);

        const items: PictureOptions[] = React.useMemo(() => [{ src: data, title: label }], [data, label]);

        const onChangeHandler = React.useCallback(
            (file: File[]) => {
                setCurrentFiles(file);

                if (onChange) {
                    onChange(file[0], type);
                }
            },
            [type, onChange],
        );

        return (
            <ImagesPreviewUpload
                className={cn(styles.preview, className)}
                images={(!data && !isLoading) || currentFiles.length ? [] : items}
                files={currentFiles}
                maxLength={1}
                isLoading={isLoading}
                onChange={onChangeHandler}
            >
                {Boolean(data) && !currentFiles.length && (
                    <ButtonFile
                        className={styles.buttonChange}
                        accept="image/*"
                        label={i18n('Change')}
                        onInputChange={onChangeHandler}
                    />
                )}

                <p className={styles.label}>{label}</p>
            </ImagesPreviewUpload>
        );
    };
