import * as React from 'react';

import { RentalRetireeDocumentPreview } from 'features/RentalRetiree/ui/RentalRetireeDocumentPreview/RentalRetireeDocumentPreview';

import {
    postUser,
    UserDocumentDriverLicenseForm,
    UserDocumentPassportForm,
    useUser,
    UseUserDocumentPhotoRes,
} from 'entities/User';
import { postUserDocumentPhoto } from 'entities/User/api/postUserDocumentPhoto/postUserDocumentPhoto';
import { UserDrivingLicenseFormSchema } from 'entities/User/types/UserDrivingLicenseFormSchema';
import { UserPassportFormSchema } from 'entities/User/types/UserPassportFormSchema';
import { UserPhotoContentTypeSchema } from 'entities/User/types/UserPhotosSchema';

import { ButtonColor } from 'shared/consts/ButtonColor';
import { ButtonSize } from 'shared/consts/ButtonSize';
import { getFetchErrorMessage } from 'shared/helpers/getFetchErrorMessage/getFetchErrorMessage';
import { useEqualDataForm } from 'shared/hooks/useEqualDataForm/useEqualDataForm';
import { useFormController } from 'shared/hooks/useFormController/useFormController';
import { FormErrorSchema } from 'shared/types/FormErrorSchema';
import { Breadcrumbs } from 'shared/ui/Breadcrumbs/Breadcrumbs';
import { Button } from 'shared/ui/Button/Button';
import { ErrorMessage } from 'shared/ui/ErrorMessage/ErrorMessage';
import { ModalContainer, ModalContainerProps } from 'shared/ui/ModalContainer/ModalContainer';

import { i18n } from 'features/RentalRetiree/ui/RentalRetireeDocumentsModal/RentalRetireeDocumentsModal.i18n';

import styles from 'features/RentalRetiree/ui/RentalRetireeDocumentsModal/RentalRetireeDocumentsModal.css';

export interface RentalRetireeDocumentsModalProps extends ModalContainerProps {
    userId: string;

    onFormSubmit(): void;
}

const EMPTY_PASSPORT_PHOTOS: UseUserDocumentPhotoRes = { pb: {}, ps: {} };
const EMPTY_DRIVE_LICENSE_PHOTOS: UseUserDocumentPhotoRes = { lf: {}, lb: {} };
const LOADING_LABEL = {
    uploading: i18n('Uploading document photos...'),
    saving: i18n('Saving documents...'),
};

export const RentalRetireeDocumentsModal: React.FC<RentalRetireeDocumentsModalProps> =
    function RentalRetireeDocumentsModal({ userId, onClose, onFormSubmit }) {
        const resource = useUser(userId);

        const [uploadFiles, setUploadFiles] = React.useState<Partial<Record<UserPhotoContentTypeSchema, File>>>({});
        const [loadingMessage, setLoadingMessage] = React.useState<Optional<'uploading' | 'saving'>>();
        const [errorMessage, setErrorMessage] = React.useState<Optional<FormErrorSchema>>();

        const documentsResource = resource.read();

        const { id, photos, documents } = documentsResource;

        const initPassportForm = documents?.passport;
        const initDriverLicenseForm = documents?.driving_license;

        const passportController = useFormController<OptionalRecord<UserPassportFormSchema>>();
        const driverLicenseController = useFormController<OptionalRecord<UserDrivingLicenseFormSchema>>();

        const passportPhotos = React.useMemo(() => ({ ...EMPTY_PASSPORT_PHOTOS, ...photos.passport }), [photos]);
        const driveLicensePhotos = React.useMemo(
            () => ({ ...EMPTY_DRIVE_LICENSE_PHOTOS, ...photos.driving_license }),
            [photos],
        );

        const hasUploadFiles = Boolean(uploadFiles && Object.keys(uploadFiles).length);

        const onPhotosChange = React.useCallback(
            (file: File, type: UserPhotoContentTypeSchema) => {
                let copyUploadPhotos = { ...uploadFiles };

                if (file) {
                    copyUploadPhotos[type] = file;
                } else if (copyUploadPhotos[type] && !file) {
                    delete copyUploadPhotos[type];
                }

                setUploadFiles(copyUploadPhotos);
            },
            [uploadFiles],
        );

        const onSubmitHandler = React.useCallback(async () => {
            if (passportController.validate() && driverLicenseController.validate()) {
                const passportData = passportController.getValues();
                const driverLicenseData = driverLicenseController.getValues();

                try {
                    if (hasUploadFiles) {
                        setLoadingMessage('uploading');
                    }

                    await postUserDocumentPhoto(uploadFiles, id);

                    setLoadingMessage('saving');

                    await postUser({
                        id,
                        passport: passportData as UserPassportFormSchema,
                        driving_license: driverLicenseData as UserDrivingLicenseFormSchema,
                    });

                    setLoadingMessage(undefined);
                    setErrorMessage(undefined);

                    // @todo: add notification
                    onFormSubmit();
                } catch (error) {
                    setUploadFiles({});
                    resource.reload();

                    setLoadingMessage(undefined);
                    setErrorMessage(getFetchErrorMessage(error));

                    return;
                }
            }
        }, [onFormSubmit, uploadFiles, passportController, driverLicenseController, resource, id, hasUploadFiles]);

        const passportForm = useEqualDataForm<UserPassportFormSchema>(initPassportForm);
        const driverLicenseForm = useEqualDataForm<UserDrivingLicenseFormSchema>(initDriverLicenseForm);

        const isEqualData = passportForm.isEqualData && driverLicenseForm.isEqualData && !hasUploadFiles;
        const buttonLabel = isEqualData ? i18n('Confirm') : i18n('Save documents and verify user');

        return (
            <ModalContainer
                className={styles.modal}
                title={
                    <>
                        <Breadcrumbs
                            className={styles.breadcrumbs}
                            items={[i18n('Starting the ride manually')]}
                        />

                        <h3 className={styles.title}>{i18n('1 of 4. Upload driver’s documents')}</h3>
                    </>
                }
                hasClose
                controls={
                    <Button
                        color={ButtonColor.PRIMARY}
                        size={ButtonSize.M}
                        label={loadingMessage ? LOADING_LABEL[loadingMessage] : buttonLabel}
                        disabled={Boolean(loadingMessage)}
                        onClick={isEqualData ? onFormSubmit : onSubmitHandler}
                    />
                }
                onClose={onClose}
            >
                <div className={styles.document}>
                    <h4 className={styles.documentTitle}>{i18n('Passport/ID card')}</h4>

                    <RentalRetireeDocumentPreview
                        className={styles.documentPreview}
                        photos={passportPhotos}
                        files={uploadFiles}
                        onChange={onPhotosChange}
                    />

                    <UserDocumentPassportForm
                        className={styles.documentForm}
                        initial={initPassportForm}
                        controller={passportController.controller}
                        onFormChange={passportForm.onFormChangeHandler}
                    />
                </div>

                <div className={styles.document}>
                    <h4 className={styles.documentTitle}>{i18n('Driver’s license')}</h4>

                    <RentalRetireeDocumentPreview
                        className={styles.documentPreview}
                        photos={driveLicensePhotos}
                        files={uploadFiles}
                        onChange={onPhotosChange}
                    />

                    <UserDocumentDriverLicenseForm
                        className={styles.documentForm}
                        initial={initDriverLicenseForm}
                        controller={driverLicenseController.controller}
                        onFormChange={driverLicenseForm.onFormChangeHandler}
                    />
                </div>

                <ErrorMessage error={errorMessage} />
            </ModalContainer>
        );
    };
