import * as React from 'react';

import { fetchRentalDamages } from 'features/RentalRetiree/api/fetchRentalDamages/fetchRentalDamages';
import { postRentalFeedback } from 'features/RentalRetiree/api/postRentalFeedback/postRentalFeedback';
import { postRentalStatus } from 'features/RentalRetiree/api/postRentalStatus/postRentalStatus';
import { RentalRetireeCarPhotos } from 'features/RentalRetiree/ui/RentalRetireeCarPhotos/RentalRetireeCarPhotos';

import { useCarOffer } from 'entities/Car/api/useCarOffer/useCarOffer';

import { ButtonColor } from 'shared/consts/ButtonColor';
import { ButtonSize } from 'shared/consts/ButtonSize';
import { dataOrEmpty } from 'shared/helpers/dataOrEmpty/dataOrEmpty';
import { formatDateTimeString } from 'shared/helpers/formatDateTimeString/formatDateTimeString';
import { formatMillisecondsToString } from 'shared/helpers/formatMillisecondsToString/formatMillisecondsToString';
import { formatNumber } from 'shared/helpers/formatNumber/formatNumber';
import { getFetchErrorMessage } from 'shared/helpers/getFetchErrorMessage/getFetchErrorMessage';
import { isValidNumber } from 'shared/helpers/isValidNumber/isValidNumber';
import { useCacheRequestContext } from 'shared/hooks/useCacheRequestContext/useCacheRequestContext';
import { FormErrorSchema } from 'shared/types/FormErrorSchema';
import { Button } from 'shared/ui/Button/Button';
import { ErrorMessage } from 'shared/ui/ErrorMessage/ErrorMessage';
import { ModalContainer, ModalContainerProps } from 'shared/ui/ModalContainer/ModalContainer';
import { Table } from 'shared/ui/Table/Table';

import { i18n } from 'features/RentalRetiree/ui/RentalRetireeFinishRideModal/RentalRetireeFinishRideModal.i18n';

import styles from 'features/RentalRetiree/ui/RentalRetireeFinishRideModal/RentalRetireeFinishRideModal.css';

export interface RentalRetireeFinishRideModalProps extends ModalContainerProps {
    offerId: string;
    userId: string;

    onFormSubmit(): void;
}

const LOADING_LABEL = {
    uploading: i18n('Uploading photos...'),
    finishing: i18n('Finishing ride...'),
};

export const RentalRetireeFinishRideModal: React.FC<RentalRetireeFinishRideModalProps> =
    function RentalRetireeFinishRideModal({ offerId, userId, onClose, onFormSubmit }) {
        const cache = useCacheRequestContext();

        let files = React.useRef<File[]>([]);

        const [loadingMessage, setLoadingMessage] = React.useState<Optional<'uploading' | 'finishing'>>();
        const [errorMessage, setErrorMessage] = React.useState<Optional<FormErrorSchema>>();

        const { data: offerData } = useCarOffer(offerId);
        const { data: damagesData, isLoading: isDamagesLoading, reload: reloadDamages } = fetchRentalDamages(offerId);

        const onClickHandler = React.useCallback(async () => {
            try {
                if (files.current.length) {
                    setLoadingMessage('uploading');
                }

                await postRentalFeedback(files.current, 'after_ride', offerId, userId);

                setLoadingMessage('finishing');

                await postRentalStatus({ session_id: offerId, tag: 'old_state_reservation' }, userId, cache);

                setLoadingMessage(undefined);
                setErrorMessage(undefined);

                onFormSubmit();

                // @todo: add notification
            } catch (error) {
                files.current = [];
                reloadDamages();

                setLoadingMessage(undefined);
                setErrorMessage(getFetchErrorMessage(error));

                return;
            }
        }, [offerId, userId, onFormSubmit, cache, files.current]);

        const onChangeHandler = React.useCallback((values: File[]) => {
            files.current = values;
        }, []);

        if (!offerData) {
            return null;
        }

        const { offer, session } = offerData;

        const hasLowFuelLevel = Boolean(
            session.current_fuel_level &&
                session.start_fuel_level &&
                session.current_fuel_level - session.start_fuel_level < 0,
        );

        const hasOvermileage = Boolean(
            session.mileage && session.mileage_limit && session.mileage > session.mileage_limit,
        );

        return (
            <ModalContainer
                className={styles.modal}
                title={i18n('Confirm the finish of the ride')}
                hasClose
                controls={
                    <Button
                        color={ButtonColor.PRIMARY}
                        size={ButtonSize.M}
                        label={loadingMessage ? LOADING_LABEL[loadingMessage] : i18n('Confirm and finish')}
                        disabled={Boolean(loadingMessage)}
                        onClick={onClickHandler}
                    />
                }
                onClose={onClose}
            >
                <Table
                    className={styles.table}
                    hasHover={false}
                >
                    <Table.Colgroup>
                        <Table.Col className={styles.tableParam} />
                        <Table.Col />
                        <Table.Col />
                    </Table.Colgroup>

                    <Table.Head>
                        <Table.Row>
                            <Table.Th>{i18n('Parameter')}</Table.Th>
                            <Table.Th>{i18n('Start')}</Table.Th>
                            <Table.Th>{i18n('Finish')}</Table.Th>
                        </Table.Row>
                    </Table.Head>

                    <Table.Body>
                        <Table.Row>
                            <Table.Td>{i18n('Date and place')}</Table.Td>
                            <Table.Td>
                                <ul>
                                    <li>{formatDateTimeString({ date: session.actual_since || session.since })}</li>
                                    <li>{offer.delivery_location.name}</li>
                                </ul>
                            </Table.Td>
                            <Table.Td>
                                <ul>
                                    <li>{formatDateTimeString({ date: session.actual_until || session.until })}</li>
                                    <li>{offer.delivery_location.name}</li>
                                    <li>
                                        {i18n('Duration: {value}', {
                                            value: formatMillisecondsToString(session.duration),
                                        })}
                                    </li>
                                </ul>
                            </Table.Td>
                        </Table.Row>

                        <Table.Row>
                            <Table.Td>{i18n('Odometer')}</Table.Td>
                            <Table.Td>
                                {dataOrEmpty(
                                    session.start_mileage,
                                    i18n('{value} km', {
                                        value: formatNumber(session.start_mileage!, { maximumFractionDigits: 2 }),
                                    }),
                                )}
                            </Table.Td>
                            <Table.Td>
                                <ul>
                                    <li>
                                        {dataOrEmpty(
                                            session.current_mileage,
                                            i18n('{value} km', {
                                                value: formatNumber(session.current_mileage!, {
                                                    maximumFractionDigits: 2,
                                                }),
                                            }),
                                        )}
                                    </li>

                                    {isValidNumber(session.mileage) && (
                                        <li>
                                            {i18n('Mileage: {value}', {
                                                value: i18n('{value} km', {
                                                    value: formatNumber(session.mileage!, {
                                                        maximumFractionDigits: 2,
                                                    }),
                                                }),
                                            })}
                                        </li>
                                    )}

                                    {isValidNumber(session.mileage_limit) && (
                                        <li>
                                            {i18n('Limit: {value}', {
                                                value: i18n('{value} km', {
                                                    value: formatNumber(session.mileage_limit!, {
                                                        maximumFractionDigits: 2,
                                                    }),
                                                }),
                                            })}
                                        </li>
                                    )}

                                    {hasOvermileage && (
                                        <li>
                                            {i18n(
                                                'Overmileage: {value}',
                                                {
                                                    value: (
                                                        <span className={styles.error}>
                                                            {i18n('{value} km', {
                                                                value: formatNumber(
                                                                    session.mileage! - session.mileage_limit!,
                                                                    {
                                                                        maximumFractionDigits: 2,
                                                                    },
                                                                ),
                                                            })}
                                                        </span>
                                                    ),
                                                },

                                                { __raw__: true },
                                            )}
                                        </li>
                                    )}
                                </ul>
                            </Table.Td>
                        </Table.Row>

                        <Table.Row>
                            <Table.Td>{i18n('Fuel')}</Table.Td>
                            <Table.Td>
                                {dataOrEmpty(session.start_fuel_level, `${session.start_fuel_level} %`)}
                            </Table.Td>
                            <Table.Td className={hasLowFuelLevel && styles.error}>
                                {dataOrEmpty(session.current_fuel_level, `${session.current_fuel_level} %`)}
                            </Table.Td>
                        </Table.Row>

                        <Table.Row>
                            <Table.Td>{i18n('Photos of car')}</Table.Td>
                            <Table.Td>
                                <RentalRetireeCarPhotos
                                    images={damagesData?.before_ride}
                                    isLoading={isDamagesLoading}
                                    isEmpty={!damagesData?.before_ride.length}
                                />
                            </Table.Td>
                            <Table.Td>
                                <RentalRetireeCarPhotos
                                    images={damagesData?.after_ride}
                                    files={files.current}
                                    isLoading={isDamagesLoading}
                                    onChange={onChangeHandler}
                                />
                            </Table.Td>
                        </Table.Row>
                    </Table.Body>
                </Table>

                <ErrorMessage error={errorMessage} />
            </ModalContainer>
        );
    };
