import * as React from 'react';
import cn from 'classnames/bind';

import { EMPTY_DATA } from 'constants/constants';

import { postRentalPayment } from 'features/RentalRetiree/api/postRentalPayment/postRentalPayment';

import { getCarOfferCurrencyName } from 'entities/Car';
import { useCarOffer } from 'entities/Car/api/useCarOffer/useCarOffer';
import { getCarOfferInsuranceType } from 'entities/Car/helpers/getCarOfferInsuranceType/getCarOfferInsuranceType';
import { getCarOfferOptions } from 'entities/Car/helpers/getCarOfferOptions/getCarOfferOptions';
import { getCarOfferStatus } from 'entities/Car/helpers/getCarOfferStatus/getCarOfferStatus';

import { ButtonColor } from 'shared/consts/ButtonColor';
import { ButtonSize } from 'shared/consts/ButtonSize';
import { dataOrEmpty } from 'shared/helpers/dataOrEmpty/dataOrEmpty';
import { formatMillisecondsToString } from 'shared/helpers/formatMillisecondsToString/formatMillisecondsToString';
import { getFetchErrorMessage } from 'shared/helpers/getFetchErrorMessage/getFetchErrorMessage';
import { useCacheRequestContext } from 'shared/hooks/useCacheRequestContext/useCacheRequestContext';
import { FormErrorSchema } from 'shared/types/FormErrorSchema';
import { Breadcrumbs } from 'shared/ui/Breadcrumbs/Breadcrumbs';
import { Button } from 'shared/ui/Button/Button';
import { ErrorMessage } from 'shared/ui/ErrorMessage/ErrorMessage';
import { ModalContainer, ModalContainerProps } from 'shared/ui/ModalContainer/ModalContainer';
import { Table } from 'shared/ui/Table/Table';

import { i18n } from 'features/RentalRetiree/ui/RentalRetireePaymentModal/RentalRetireePaymentModal.i18n';

import styles from 'features/RentalRetiree/ui/RentalRetireePaymentModal/RentalRetireePaymentModal.css';

export interface RentalRetireePaymentModalProps extends ModalContainerProps {
    offerId: string;
    tagId: string;
    userId: string;

    onFormSubmit(): void;
}

const cx = cn.bind(styles);

export const RentalRetireePaymentModal: React.FC<RentalRetireePaymentModalProps> = function RentalRetireePaymentModal({
    offerId,
    tagId,
    userId,
    onClose,
    onFormSubmit,
}) {
    const cache = useCacheRequestContext();

    const [loadingMessage, setLoadingMessage] = React.useState<boolean>();
    const [errorMessage, setErrorMessage] = React.useState<Optional<FormErrorSchema>>();

    const { data, reload } = useCarOffer(offerId);

    const onSubmitHandler = React.useCallback(async () => {
        try {
            setLoadingMessage(true);

            await postRentalPayment(data, offerId, tagId, userId, cache);

            setLoadingMessage(undefined);
            setErrorMessage(undefined);

            onFormSubmit();

            // @todo: add notification
        } catch (error) {
            // try to fix current data state
            // in case of postRentalPayment done partially state update
            reload();

            setLoadingMessage(undefined);
            setErrorMessage(getFetchErrorMessage(error));

            return;
        }
    }, [offerId, tagId, userId, onFormSubmit, cache, data]);

    if (!data) {
        return null;
    }

    const { offer, offerConfig, session } = data;

    const isPaid = data.offer.status === 'paid';
    const currencyName = getCarOfferCurrencyName(offer.currency, offerConfig.currencies);
    const offerOptions = getCarOfferOptions(offer.offer_options, offerConfig.offer_options);
    const offerStatus = getCarOfferStatus(offer.status, offerConfig.statuses);

    const buttonLabel = isPaid ? i18n('Start inspection') : i18n('Mark as paid and start inspection');

    return (
        <ModalContainer
            className={styles.modal}
            title={
                <>
                    <Breadcrumbs
                        className={styles.breadcrumbs}
                        items={[i18n('Starting the ride manually')]}
                    />

                    <h3 className={styles.title}>
                        {isPaid ? i18n('3 of 4. Payment') : i18n('3 of 4. Check the payment')}
                    </h3>
                </>
            }
            hasClose
            controls={
                <Button
                    color={ButtonColor.PRIMARY}
                    size={ButtonSize.M}
                    label={loadingMessage ? i18n('Starting inspection...') : buttonLabel}
                    disabled={loadingMessage}
                    onClick={onSubmitHandler}
                />
            }
            onClose={onClose}
        >
            <Table
                className={styles.table}
                hasHover={false}
            >
                <Table.Colgroup>
                    <Table.Col className={styles.tableParam} />
                    <Table.Col />
                </Table.Colgroup>

                <Table.Body>
                    <Table.Row>
                        <Table.Td>
                            {i18n('Total for {value}', { value: formatMillisecondsToString(session.duration) })}
                        </Table.Td>
                        <Table.Td>
                            {dataOrEmpty(offer.total_payment, `${offer.total_payment} ${currencyName}`)}
                        </Table.Td>
                    </Table.Row>

                    <Table.Row>
                        <Table.Td>{i18n('Deposit')}</Table.Td>
                        <Table.Td>{dataOrEmpty(offer.deposit, `${offer.deposit} ${currencyName}`)}</Table.Td>
                    </Table.Row>

                    <Table.Row>
                        <Table.Td>{i18n('Insurance')}</Table.Td>
                        <Table.Td>
                            {getCarOfferInsuranceType(offer.insurance_type, offerConfig.insurance_types)}
                        </Table.Td>
                    </Table.Row>

                    <Table.Row>
                        <Table.Td>{i18n('Options')}</Table.Td>
                        <Table.Td>
                            {offerOptions.length ? (
                                <ul>
                                    {offerOptions.map((item, key) => (
                                        <li key={key}>{item}</li>
                                    ))}
                                </ul>
                            ) : (
                                EMPTY_DATA
                            )}
                        </Table.Td>
                    </Table.Row>

                    <Table.Row>
                        <Table.Td>{i18n('Status')}</Table.Td>
                        <Table.Td>
                            <span className={cx(styles.status, { green: isPaid })}>
                                {isPaid ? offerStatus : i18n('{status} unpaid', { status: offerStatus })}
                            </span>
                        </Table.Td>
                    </Table.Row>
                </Table.Body>
            </Table>

            <ErrorMessage error={errorMessage} />
        </ModalContainer>
    );
};
