import * as React from 'react';

import { fetchRentalDamages } from 'features/RentalRetiree/api/fetchRentalDamages/fetchRentalDamages';
import { postRentalFeedback } from 'features/RentalRetiree/api/postRentalFeedback/postRentalFeedback';
import { postRentalStatus } from 'features/RentalRetiree/api/postRentalStatus/postRentalStatus';
import { RentalRetireeCarPhotos } from 'features/RentalRetiree/ui/RentalRetireeCarPhotos/RentalRetireeCarPhotos';

import { useCarOffer } from 'entities/Car/api/useCarOffer/useCarOffer';

import { ButtonColor } from 'shared/consts/ButtonColor';
import { ButtonSize } from 'shared/consts/ButtonSize';
import { dataOrEmpty } from 'shared/helpers/dataOrEmpty/dataOrEmpty';
import { formatDateTimeString } from 'shared/helpers/formatDateTimeString/formatDateTimeString';
import { formatNumber } from 'shared/helpers/formatNumber/formatNumber';
import { getFetchErrorMessage } from 'shared/helpers/getFetchErrorMessage/getFetchErrorMessage';
import { useCacheRequestContext } from 'shared/hooks/useCacheRequestContext/useCacheRequestContext';
import { FormErrorSchema } from 'shared/types/FormErrorSchema';
import { Breadcrumbs } from 'shared/ui/Breadcrumbs/Breadcrumbs';
import { Button } from 'shared/ui/Button/Button';
import { ErrorMessage } from 'shared/ui/ErrorMessage/ErrorMessage';
import { ModalContainer, ModalContainerProps } from 'shared/ui/ModalContainer/ModalContainer';
import { Table } from 'shared/ui/Table/Table';

import { i18n } from 'features/RentalRetiree/ui/RentalRetireeStartRideModal/RentalRetireeStartRideModal.i18n';

import styles from 'features/RentalRetiree/ui/RentalRetireeStartRideModal/RentalRetireeStartRideModal.css';

export interface RentalRetireeStartRideModalProps extends ModalContainerProps {
    offerId: string;
    userId: string;

    onFormSubmit(): void;
}

const LOADING_LABEL = {
    uploading: i18n('Uploading photos...'),
    starting: i18n('Starting ride...'),
};

export const RentalRetireeStartRideModal: React.FC<RentalRetireeStartRideModalProps> =
    function RentalRetireeStartRideModal({ offerId, userId, onClose, onFormSubmit }) {
        const cache = useCacheRequestContext();

        let files = React.useRef<File[]>([]);

        const [loadingMessage, setLoadingMessage] = React.useState<Optional<'uploading' | 'starting'>>();
        const [errorMessage, setErrorMessage] = React.useState<Optional<FormErrorSchema>>();

        const { data: offerData } = useCarOffer(offerId);
        const { data: damagesData, isLoading: isDamagesLoading, reload: reloadDamages } = fetchRentalDamages(offerId);

        const onClickHandler = React.useCallback(async () => {
            try {
                if (files.current.length) {
                    setLoadingMessage('uploading');
                }

                await postRentalFeedback(files.current, 'before_ride', offerId, userId);

                setLoadingMessage('starting');

                await postRentalStatus({ session_id: offerId, tag: 'old_state_riding' }, userId, cache);

                setLoadingMessage(undefined);
                setErrorMessage(undefined);

                onFormSubmit();

                // @todo: add notification
            } catch (error) {
                files.current = [];
                reloadDamages();

                setLoadingMessage(undefined);
                setErrorMessage(getFetchErrorMessage(error));

                return;
            }
        }, [offerId, userId, onFormSubmit, cache, files.current]);

        const onChangeHandler = React.useCallback((values: File[]) => {
            files.current = values;
        }, []);

        if (!offerData) {
            return null;
        }

        const { offer, session } = offerData;

        return (
            <ModalContainer
                className={styles.modal}
                title={
                    <>
                        <Breadcrumbs
                            className={styles.breadcrumbs}
                            items={[i18n('Starting the ride manually')]}
                        />

                        <h3 className={styles.title}>
                            {i18n('4 of 4. Check the data, upload the car photos and start the ride')}
                        </h3>
                    </>
                }
                hasClose
                controls={
                    <Button
                        color={ButtonColor.PRIMARY}
                        size={ButtonSize.M}
                        label={loadingMessage ? LOADING_LABEL[loadingMessage] : i18n('Start ride')}
                        disabled={Boolean(loadingMessage)}
                        onClick={onClickHandler}
                    />
                }
                onClose={onClose}
            >
                <Table
                    className={styles.table}
                    hasHover={false}
                >
                    <Table.Colgroup>
                        <Table.Col className={styles.tableParam} />
                        <Table.Col />
                    </Table.Colgroup>

                    <Table.Body>
                        <Table.Row>
                            <Table.Td>{i18n('Date and place')}</Table.Td>
                            <Table.Td>
                                <ul>
                                    <li>{formatDateTimeString({ date: session.since })}</li>
                                    <li>{offer.delivery_location.name}</li>
                                </ul>
                            </Table.Td>
                        </Table.Row>

                        <Table.Row>
                            <Table.Td>{i18n('Odometer')}</Table.Td>
                            <Table.Td>
                                {dataOrEmpty(
                                    session.start_mileage,
                                    i18n('{value} km', {
                                        value: formatNumber(session.start_mileage!, { maximumFractionDigits: 2 }),
                                    }),
                                )}
                            </Table.Td>
                        </Table.Row>

                        <Table.Row>
                            <Table.Td>{i18n('Fuel')}</Table.Td>
                            <Table.Td>
                                {dataOrEmpty(session.start_fuel_level, `${session.start_fuel_level} %`)}
                            </Table.Td>
                        </Table.Row>

                        <Table.Row>
                            <Table.Td>{i18n('Photos of car')}</Table.Td>
                            <Table.Td>
                                <RentalRetireeCarPhotos
                                    images={damagesData?.before_ride}
                                    files={files.current}
                                    isLoading={isDamagesLoading}
                                    onChange={onChangeHandler}
                                />
                            </Table.Td>
                        </Table.Row>
                    </Table.Body>
                </Table>

                <ErrorMessage error={errorMessage} />
            </ModalContainer>
        );
    };
