import * as React from 'react';
import cn from 'classnames';

import { postSettingsRatesForm } from 'features/SettingsDailyRates/api/postSettingsRatesForm/postSettingsRatesForm';
import { SettingsRatesFormSchema } from 'features/SettingsDailyRates/types/SettingsRatesFormSchema';
import { SettingsDailyRatesForm } from 'features/SettingsDailyRates/ui/SettingsDailyRatesForm/SettingsDailyRatesForm';

import { UseCarOfferPricingResource } from 'entities/Car/api/useCarOfferPricing/useCarOfferPricing';

import { getFetchErrorMessage } from 'shared/helpers/getFetchErrorMessage/getFetchErrorMessage';
import { resourceCallbackReader } from 'shared/helpers/resourceCallbackReader/resourceCallbackReader';
import { useEqualDataForm } from 'shared/hooks/useEqualDataForm/useEqualDataForm';
import { useFormController } from 'shared/hooks/useFormController/useFormController';

import { i18n } from 'features/SettingsDailyRates/ui/SettingsDailyRates/SettingsDailyRates.i18n';

import styles from 'features/SettingsDailyRates/ui/SettingsDailyRates/SettingsDailyRates.css';

export interface SettingsDailyRatesProps {
    className?: string;
    resource: UseCarOfferPricingResource;
}

export const SettingsDailyRates: React.FC<SettingsDailyRatesProps> = function SettingsDailyRates({
    className,
    resource,
}) {
    const { ranges, rates, ratesGrid, currency } = resource.read();

    const { isEqualData, onFormChangeHandler } = useEqualDataForm<SettingsRatesFormSchema>(ratesGrid);
    const { getValues, validate, setError, controller } = useFormController<OptionalRecord<SettingsRatesFormSchema>>();

    const onInputBlurHandler = React.useCallback(async () => {
        if (validate() && !isEqualData) {
            const formData = getValues();

            const { report } = await resourceCallbackReader(resource);

            try {
                await postSettingsRatesForm(formData as SettingsRatesFormSchema, report);

                // @todo: add notification

                resource.reload();
            } catch (error) {
                setError('_serverError', getFetchErrorMessage(error));
            }
        }
    }, [resource, isEqualData, getValues, validate, setError]);

    if (!ratesGrid || !Object.keys(ratesGrid).length) {
        return (
            <div className={cn(styles.rates, styles.empty, className)}>
                <p className={styles.emptyText}>{i18n('No price categories found')}</p>
            </div>
        );
    }

    return (
        <div className={cn(styles.rates, className)}>
            <SettingsDailyRatesForm
                className={styles.rates}
                title={i18n('Default rates')}
                description={i18n('Common rule for the whole year')}
                rows={rates}
                ranges={ranges}
                currency={currency}
                initial={ratesGrid}
                controller={controller}
                onFormChange={onFormChangeHandler}
                onInputBlur={onInputBlurHandler}
            />
        </div>
    );
};
