import * as React from 'react';
import cn from 'classnames/bind';

import { SettingsRatesFormSchema } from 'features/SettingsDailyRates/types/SettingsRatesFormSchema';

import { CAR_OFFER_PRICING_SEP } from 'entities/Car/consts/constants';

import { InputSize } from 'shared/consts/InputSize';
import { useForm } from 'shared/hooks/useForm/useForm';
import { UseFormControllerSub } from 'shared/hooks/useFormController/useFormController';
import { ErrorMessage } from 'shared/ui/ErrorMessage/ErrorMessage';
import { Input, InputProps } from 'shared/ui/Input/Input';
import { Table } from 'shared/ui/Table/Table';

import { i18n } from 'features/SettingsDailyRates/ui/SettingsDailyRatesForm/SettingsDailyRatesForm.i18n';

import styles from 'features/SettingsDailyRates/ui/SettingsDailyRatesForm/SettingsDailyRatesForm.css';

export interface SettingsDailyRatesFormProps extends Pick<InputProps, 'onBlur'> {
    className?: string;

    title: string;
    description: string;
    rows: Array<{ id: string; label: string; info?: React.ReactNode }>;
    ranges: Dict;
    currency?: string;

    initial?: SettingsRatesFormSchema;
    controller?: UseFormControllerSub<SettingsRatesFormSchema>;

    onFormChange?(state: SettingsRatesFormSchema): void;

    onInputBlur?(): void;
}

const INIT_FORM: OptionalRecord<SettingsRatesFormSchema> = {};

const cx = cn.bind(styles);

export const SettingsDailyRatesForm: React.FC<SettingsDailyRatesFormProps> = function SettingsDailyRatesForm({
    className,
    title,
    description,
    rows,
    ranges,
    currency,
    initial,
    controller,
    onFormChange,
    onInputBlur,
}) {
    const { update, watch, errors } = useForm<OptionalRecord<SettingsRatesFormSchema>>({
        init: { ...INIT_FORM, ...initial },
        validation: {},
        controller,
        onFormChange,
    });

    const columns = React.useMemo(() => Object.keys(ranges), [ranges]);

    return (
        <div className={cn(styles.grid, className)}>
            <div className={styles.header}>
                <h3 className={styles.title}>{title}</h3>
                <p className={styles.description}>{description}</p>
            </div>

            <div className={cn(styles.body, className)}>
                <Table className={styles.table}>
                    <Table.Colgroup>
                        <Table.Col gap />
                        <Table.Col className={styles.colLabel} />

                        {columns.map((index) => (
                            <Table.Col
                                className={styles.colInput}
                                key={index}
                            />
                        ))}

                        <Table.Col />
                        <Table.Col gap />
                    </Table.Colgroup>

                    <Table.Head>
                        <Table.Row>
                            <Table.Th gap />
                            <Table.Th>{i18n('Category')}</Table.Th>

                            {columns.map((item, index) => (
                                <Table.Th
                                    className={cx({ last: index === columns.length - 1 })}
                                    key={item}
                                >
                                    {currency ? `${ranges[item]}, ${currency}` : ranges[item]}
                                </Table.Th>
                            ))}

                            <Table.Th />
                            <Table.Th gap />
                        </Table.Row>
                    </Table.Head>

                    <Table.Body>
                        {rows.map(({ id, label, info }) => (
                            <Table.Row key={id}>
                                <Table.Td gap />

                                <Table.Td>
                                    <span className={styles.rowLabel}>{label}</span>

                                    {info && <span className={styles.rowInfo}>{info}</span>}
                                </Table.Td>

                                {columns.map((item, index) => {
                                    const key = item + CAR_OFFER_PRICING_SEP + id;

                                    return (
                                        <Table.Td
                                            className={cx({ last: index === columns.length - 1 })}
                                            key={item}
                                        >
                                            <Input
                                                className={styles.tableInput}
                                                inputSize={InputSize.M}
                                                inputMode="numeric"
                                                type="number"
                                                value={watch(key)}
                                                hasError={Boolean(errors[key])}
                                                withBackground
                                                onInputChange={update(key)}
                                                onBlur={onInputBlur}
                                            />
                                        </Table.Td>
                                    );
                                })}

                                <Table.Td />
                                <Table.Td gap />
                            </Table.Row>
                        ))}
                    </Table.Body>
                </Table>
            </div>

            <ErrorMessage
                className={styles.error}
                error={errors._serverError}
            />
        </div>
    );
};
