import * as React from 'react';
import { useHistory, useRouteMatch } from 'react-router-dom';
import cn from 'classnames';

import { isSettingsPricingFlag } from 'utils/isSettingsPricingFlag';
import { isShowNotificationPageFlag } from 'utils/isShowNotificationPage';

import { Path } from 'shared/consts/Path';
import { TabSize } from 'shared/consts/TabSize';
import { generateRouterPath } from 'shared/helpers/generateRouterPath/generateRouterPath';
import { TabItemOptions } from 'shared/ui/TabItem/TabItem';
import { Tabs } from 'shared/ui/Tabs/Tabs';

import { i18n } from 'features/SettingsHeaderTabs/ui/SettingsHeaderTabs/SettingsHeaderTabs.i18n';

import styles from 'features/SettingsHeaderTabs/ui/SettingsHeaderTabs/SettingsHeaderTabs.css';

enum SettingsHeaderTabsValue {
    GENERAL = 'general',
    PRICING = 'pricing',
    NOTIFICATIONS = 'notifications',
}

export interface SettingsHeaderTabsProps {
    className?: string;
}

export const SettingsHeaderTabs: React.FC<SettingsHeaderTabsProps> = function SettingsHeaderTabs({ className }) {
    const history = useHistory();
    const routeMatch = useRouteMatch<{ tab: string }>(`${Path.SETTINGS}/:tab`);

    const [currentItem, setCurrentItem] = React.useState<string>();

    React.useEffect(() => {
        setCurrentItem(routeMatch?.params?.tab || SettingsHeaderTabsValue.GENERAL);
    }, [routeMatch]);

    const onItemClickHandler = React.useCallback(
        (value: SettingsHeaderTabsValue) => {
            setCurrentItem(value);

            switch (value) {
                case SettingsHeaderTabsValue.GENERAL:
                    history.push(generateRouterPath(Path.SETTINGS_GENERAL));
                    break;

                case SettingsHeaderTabsValue.PRICING:
                    history.push(generateRouterPath(Path.SETTINGS_PRICING));
                    break;

                case SettingsHeaderTabsValue.NOTIFICATIONS:
                    history.push(generateRouterPath(Path.SETTINGS_NOTIFICATIONS));
                    break;
            }
        },
        [history],
    );

    const items = React.useMemo(
        (): TabItemOptions[] => [
            { label: i18n('General'), value: SettingsHeaderTabsValue.GENERAL },
            ...(isSettingsPricingFlag() ? [{ label: i18n('Pricing'), value: SettingsHeaderTabsValue.PRICING }] : []),
            ...(isShowNotificationPageFlag()
                ? [{ label: i18n('Notifications'), value: SettingsHeaderTabsValue.NOTIFICATIONS }]
                : []),
        ],
        [],
    );

    if (items.length === 1) {
        return null;
    }

    return (
        <Tabs
            className={cn(styles.settingsHeaderTabs, className)}
            size={TabSize.M}
            items={items}
            currentItem={currentItem}
            onItemClick={onItemClickHandler}
        />
    );
};
