import * as React from 'react';
import cn from 'classnames';

import { SignalDetailsMediaPlayer } from 'features/SignalDetails/ui/SignalDetailsMediaPlayer/SignalDetailsMediaPlayer';

import { SignalMediaDetailsSchema } from 'entities/Signal/types/SignalMediaDetailsSchema';

import styles from 'features/SignalDetails/ui/SignalDetailsMedia/SignalDetailsMedia.css';

export interface SignalDetailsMediaProps {
    className?: string;
    media: SignalMediaDetailsSchema;
}

export const SignalDetailsMedia: React.FC<SignalDetailsMediaProps> = React.memo(function SignalDetailsMedia({
    className,
    media,
}) {
    const internalVideo = React.useRef<Nullable<HTMLVideoElement>>(null);
    const externalVideo = React.useRef<Nullable<HTMLVideoElement>>(null);

    const onPlay = React.useCallback(() => {
        const internalElem = internalVideo.current;

        if (internalElem && internalElem.paused) {
            internalElem.play();
        }
    }, [internalVideo]);

    const onPause = React.useCallback(() => {
        const internalElem = internalVideo.current;

        if (internalElem && !internalElem.paused) {
            internalElem.pause();
        }
    }, [internalVideo]);

    const onSeeking = React.useCallback(() => {
        const internalElem = internalVideo.current;
        const externalElem = externalVideo.current;

        if (internalElem && externalElem && internalElem.currentTime !== externalElem.currentTime) {
            internalElem.currentTime = externalElem.currentTime;
        }
    }, [internalVideo, externalVideo]);

    const duoVideoMode = Boolean(media.video && media.external_video);
    const syncEvents = duoVideoMode
        ? {
              onPlay,
              onPlaying: onSeeking,
              onPause,
              onSeeking,
              onSeeked: onSeeking,
          }
        : {};

    return (
        <div className={cn(styles.media, className)}>
            <SignalDetailsMediaPlayer
                ref={internalVideo}
                className={cn(styles.mediaFront, duoVideoMode ? styles.mediaMini : undefined)}
                video={media.video}
                photo={media.photo}
                autoPlay={!duoVideoMode}
                controls
                controlsList={duoVideoMode ? 'nofullscreen' : undefined}
            />

            <SignalDetailsMediaPlayer
                ref={externalVideo}
                className={styles.mediaExternal}
                video={media.external_video}
                photo={media.external_photo}
                autoPlay
                controls
                controlsList={duoVideoMode ? 'nofullscreen' : undefined}
                {...syncEvents}
            />
        </div>
    );
});
