import * as React from 'react';
import { useHistory, useLocation } from 'react-router-dom';
import cn from 'classnames/bind';

import { MODAL_OBJECT_ID_CGI, MODAL_OBJECT_TYPE_CGI } from 'constants/constants';

import { durationUTCBetween } from 'utils/date/durationUTCBetween';
import shortDateTime from 'utils/date/shortDateTime';
import { isCarSignalResolutionFlag } from 'utils/isCarSignalResolutionFlag';
import { isSignalsSupportFlag } from 'utils/isSignalsSupportFlag';

import { SignalDetailsMedia } from 'features/SignalDetails/ui/SignalDetailsMedia/SignalDetailsMedia';

import { formatCarModel, getCarNumber } from 'entities/Car';
import { postApproveSignal } from 'entities/Signal/api/postApproveSignal/postApproveSignal';
import { postResolutionSignal } from 'entities/Signal/api/postResolutionSignal/postResolutionSignal';
import { UseSignalListServerData } from 'entities/Signal/api/useSignalList/useSignalList';
import { SignalResolution } from 'entities/Signal/consts/SignalResolution';
import { getSignalTypeTitle } from 'entities/Signal/helpers/getSignalTypeTitle/getSignalTypeTitle';
import { SignalFormattedSchema } from 'entities/Signal/types/SignalFormattedSchema';
import { SignalMediaDetailsSchema } from 'entities/Signal/types/SignalMediaDetailsSchema';

import { ButtonColor } from 'shared/consts/ButtonColor';
import { ButtonSize } from 'shared/consts/ButtonSize';
import { Path } from 'shared/consts/Path';
import { generateRouterPath } from 'shared/helpers/generateRouterPath/generateRouterPath';
import { Button } from 'shared/ui/Button/Button';
import { DetailsContainer, DetailsContainerItemOptions } from 'shared/ui/DetailsContainer/DetailsContainer';
import { Link } from 'shared/ui/Link/Link';
import { ModalContainer } from 'shared/ui/ModalContainer/ModalContainer';
import { TextWithDot } from 'shared/ui/TextWithDot/TextWithDot';

import { SessionTime } from 'components/Cars/CarCard/CarSessionsTable/sessionTime';
import { SCORING_TRACE_TAG } from 'components/Cars/constants';
import NotificationCenterContext, { addNotification } from 'components/NotificationCenter/store';
import { NotificationIconType } from 'components/NotificationCenter/types';
import { ModalObjectTypes } from 'components/types';
import { DurationUTCBetween } from 'components/ui/DurationUTCBetween';

import { i18n } from 'features/SignalDetails/ui/SignalDetailsModal/SignalDetailsModal.i18n';

import styles from 'features/SignalDetails/ui/SignalDetailsModal/SignalDetailsModal.css';

export interface SignalDetailsModalProps {
    className?: string;

    signal?: SignalFormattedSchema;
    onClose: () => void;
    onNextClick?: () => void;
    fetchCustomPage?: (data: UseSignalListServerData) => void;
    onRemoveData?: (tag_id: string) => void;
    onUpdateData?: (tag_id: string, newData: Partial<SignalFormattedSchema>) => void;
}

const cx = cn.bind(styles);

const RESOLUTIONS: Record<SignalResolution, string> = {
    [SignalResolution.DELETE]: i18n('Delete'),
    [SignalResolution.HIDE]: i18n('Hide'),
    [SignalResolution.WRONG_EVENT]: i18n('Wrong event'),
    [SignalResolution.WRONG_DRIVER]: i18n('Wrong driver'),
};

export const SignalDetailsModal: React.FC<SignalDetailsModalProps> = React.memo(function SignalDetailsModal({
    className,
    signal,
    onClose,
    onNextClick,
    onRemoveData,
    onUpdateData,
    children,
}) {
    const location = useLocation();
    const history = useHistory();

    const { name, since, until, description, is_actual, details, trace, car: signalCar, tag_id } = signal || {};
    const { priority, display_name, type, is_permanent } = description || {};
    const time = since && shortDateTime(since);

    const { details: traceDetails, object_id: traceId } = trace || {};
    const { since: traceSince, until: traceUntil } = traceDetails || {};

    const { details: signalCarDetails, object_id: carId } = signalCar || {};
    const { number, model } = signalCarDetails || {};

    const media: SignalMediaDetailsSchema = details && details.media;
    const resolution: SignalResolution = details && details.resolution;
    const signalTypeTitle = getSignalTypeTitle(type);

    const [changeResolution, setChangeResolution] = React.useState<boolean>(false);
    const [localResolution, setLocalResolution] = React.useState<Optional<SignalResolution>>(resolution);

    const showResolutionButtons =
        !isSignalsSupportFlag() && isCarSignalResolutionFlag() && media && (!localResolution || changeResolution);

    const { notificationDispatch } = React.useContext(NotificationCenterContext) || {};

    const priorityText = is_actual
        ? priority === 'critical'
            ? i18n('Critical and actual')
            : i18n('Normal and actual')
        : priority === 'critical'
        ? i18n('Critical')
        : i18n('Normal');

    const onTraceClick = React.useCallback(() => {
        let searchParams = new URLSearchParams(location.search);
        searchParams.set(MODAL_OBJECT_TYPE_CGI, ModalObjectTypes.SESSION);
        searchParams.set(MODAL_OBJECT_ID_CGI, traceId ?? '');
        history.push(`${location.pathname}?${searchParams}`);
    }, [history, location.pathname, location.search, traceId]);

    const removeItem = React.useCallback(() => {
        if (tag_id && onRemoveData) {
            onRemoveData(tag_id);

            if (onNextClick) {
                onNextClick();
            }
        }
    }, [onNextClick, onRemoveData, tag_id]);

    const onApproveClick = React.useCallback(() => {
        if (!tag_id) {
            return;
        }

        postApproveSignal({
            tag_id,
        })
            .ready()
            .then(() => {
                removeItem();

                notificationDispatch(
                    addNotification({
                        title: i18n('Approved'),
                    }),
                );
            })
            .catch(() => {
                notificationDispatch(
                    addNotification({
                        title: i18n('Failed to approve'),
                        iconType: NotificationIconType.ERROR,
                    }),
                );
            });
    }, [tag_id, removeItem, notificationDispatch]);

    const setResolution = React.useCallback(
        (resolution: SignalResolution) => {
            if (!tag_id) {
                return;
            }

            postResolutionSignal({
                tag_id,
                resolution,
            })
                .ready()
                .then(() => {
                    if (onUpdateData) {
                        onUpdateData(tag_id, {
                            details: {
                                ...details,
                                resolution,
                            },
                        });
                    }

                    if (resolution === SignalResolution.DELETE) {
                        removeItem();
                    }

                    setLocalResolution(resolution);
                    setChangeResolution(false);
                })
                .catch(() => {
                    notificationDispatch(
                        addNotification({
                            title: isSignalsSupportFlag() ? i18n('Failed to deny') : i18n('Failed to set resolution'),
                            iconType: NotificationIconType.ERROR,
                        }),
                    );
                });
        },
        [details, notificationDispatch, onUpdateData, removeItem, tag_id],
    );

    const onHideClick = React.useCallback(() => {
        setResolution(SignalResolution.HIDE);
    }, [setResolution]);

    const onDeleteClick = React.useCallback(() => {
        setResolution(SignalResolution.DELETE);
    }, [setResolution]);

    const onWrongEventClick = React.useCallback(() => {
        setResolution(SignalResolution.WRONG_EVENT);
    }, [setResolution]);

    const onChangeClick = React.useCallback(() => {
        setChangeResolution(true);
    }, []);

    const detailsItems: DetailsContainerItemOptions[] = React.useMemo(() => {
        if (!since || !priority) {
            return [];
        }

        return [
            {
                id: 'type',
                label: i18n('Type'),
                value: signalTypeTitle,
            },

            {
                id: 'level',
                label: i18n('Level'),
                value: (
                    <TextWithDot
                        className={cx(styles[priority])}
                        pulsar={is_actual && priority === 'critical'}
                    >
                        {priorityText}
                    </TextWithDot>
                ),
            },

            {
                id: 'duration',
                label: i18n('Duration'),
                value:
                    !is_permanent && name !== SCORING_TRACE_TAG ? (
                        <DurationUTCBetween duration={durationUTCBetween(since, until)} />
                    ) : null,
            },

            {
                id: 'ride',
                label: i18n('Ride'),
                value: traceSince && (
                    <Link onClick={onTraceClick}>
                        <SessionTime
                            start={traceSince}
                            finish={traceUntil}
                        />
                    </Link>
                ),
            },

            {
                id: 'car',
                label: i18n('Car'),
                value: carId && (
                    <Link href={generateRouterPath(Path.CAR, { id: carId })}>
                        {formatCarModel(model)} {getCarNumber(number)}
                    </Link>
                ),
            },

            {
                id: 'resolution',
                label: i18n('Resolution'),
                value:
                    localResolution && !changeResolution ? (
                        <>
                            <span>{RESOLUTIONS[localResolution]}.</span>
                            <Link
                                onClick={onChangeClick}
                                className={styles.change}
                            >
                                {i18n('Change')}
                            </Link>
                        </>
                    ) : null,
            },
        ];
    }, [
        since,
        priority,
        signalTypeTitle,
        is_actual,
        priorityText,
        is_permanent,
        name,
        until,
        traceSince,
        onTraceClick,
        traceUntil,
        carId,
        model,
        number,
        localResolution,
        changeResolution,
        onChangeClick,
    ]);

    return (
        <ModalContainer
            modalContainerClassName={styles.modalContainer}
            className={cx(styles.content, { wide: Boolean(media) }, [className])}
            onClose={onClose}
            hasClose
        >
            {Boolean(signal) && (
                <div className={styles.signalContent}>
                    {media && <SignalDetailsMedia media={media} />}

                    <div className={styles.details}>
                        <h3 className={styles.title}>
                            {time ? (
                                <>
                                    {display_name} — {time}
                                </>
                            ) : (
                                display_name
                            )}
                        </h3>

                        <DetailsContainer
                            className={styles.detailsItems}
                            items={detailsItems}
                        />

                        {showResolutionButtons && (
                            <div className={styles.resolutionButtons}>
                                <Button
                                    color={ButtonColor.SECONDARY}
                                    size={ButtonSize.M}
                                    label={i18n('Hide')}
                                    onClick={onHideClick}
                                />
                                <Button
                                    color={ButtonColor.SECONDARY}
                                    size={ButtonSize.M}
                                    label={i18n('Delete')}
                                    onClick={onDeleteClick}
                                />
                                <Button
                                    color={ButtonColor.SECONDARY}
                                    size={ButtonSize.M}
                                    label={i18n('Wrong event')}
                                    onClick={onWrongEventClick}
                                />
                            </div>
                        )}

                        {isSignalsSupportFlag() && (
                            <div className={styles.approveButton}>
                                <Button
                                    color={ButtonColor.PRIMARY}
                                    size={ButtonSize.M}
                                    label={i18n('Approve')}
                                    onClick={onApproveClick}
                                />
                                <Button
                                    color={ButtonColor.SECONDARY}
                                    size={ButtonSize.M}
                                    label={i18n('Deny')}
                                    onClick={onDeleteClick}
                                />
                            </div>
                        )}
                    </div>
                </div>
            )}

            {children}
        </ModalContainer>
    );
});
