import * as React from 'react';
import { useHistory, useRouteMatch } from 'react-router-dom';
import cn from 'classnames';

import { showZones } from 'utils/showZones';

import { Path } from 'shared/consts/Path';
import { TabSize } from 'shared/consts/TabSize';
import { TabItemOptions } from 'shared/ui/TabItem/TabItem';
import { Tabs } from 'shared/ui/Tabs/Tabs';

import { i18n } from 'features/SignalsHeaderTabs/ui/SignalsHeaderTabs/SignalsHeaderTabs.i18n';

import styles from 'features/SignalsHeaderTabs/ui/SignalsHeaderTabs/SignalsHeaderTabs.css';

enum SignalsHeaderTabsValue {
    SIGNALS = 'signals',
    MANAGE = 'manage',
    ZONES = 'zones',
}

export interface SignalsHeaderTabsProps {
    className?: string;
}

export const SignalsHeaderTabs: React.FC<SignalsHeaderTabsProps> = function SignalsHeaderTabs({ className }) {
    const history = useHistory();
    const routeMatch = useRouteMatch<{ tab: string }>(`${Path.SIGNALS}/:tab`);

    const [currentItem, setCurrentItem] = React.useState<string>();

    React.useEffect(() => {
        setCurrentItem(routeMatch?.params?.tab || SignalsHeaderTabsValue.SIGNALS);
    }, [routeMatch]);

    const onItemClickHandler = React.useCallback((value: SignalsHeaderTabsValue) => {
        setCurrentItem(value);

        switch (value) {
            case SignalsHeaderTabsValue.SIGNALS:
                history.push(Path.SIGNALS);
                break;

            case SignalsHeaderTabsValue.MANAGE:
                history.push(Path.SIGNALS_MANAGE);
                break;

            case SignalsHeaderTabsValue.ZONES:
                history.push(Path.SIGNALS_ZONES);
                break;
        }
    }, []);

    const hasZones = showZones();

    const items = React.useMemo(() => {
        let list: TabItemOptions[] = [
            { label: i18n('Signals'), value: SignalsHeaderTabsValue.SIGNALS },
            { label: i18n('Settings'), value: SignalsHeaderTabsValue.MANAGE },
        ];

        if (hasZones) {
            list.push({ label: i18n('Geofences'), value: SignalsHeaderTabsValue.ZONES });
        }

        return list;
    }, [hasZones]);

    return (
        <Tabs
            className={cn(styles.signalsHeaderTabs, className)}
            size={TabSize.M}
            items={items}
            currentItem={currentItem}
            onItemClick={onItemClickHandler}
        />
    );
};
