import * as React from 'react';
import cn from 'classnames';

import { SignalsTableRow } from 'features/SignalsTable/ui/SignalsTableRow/SignalsTableRow';

import { SignalFormattedSchema } from 'entities/Signal/types/SignalFormattedSchema';

import { Link } from 'shared/ui/Link/Link';
import { Spin } from 'shared/ui/Spin/Spin';
import { Table } from 'shared/ui/Table/Table';

import { i18n } from 'features/SignalsTable/ui/SignalsTable/SignalsTable.i18n';

import styles from 'features/SignalsTable/ui/SignalsTable/SignalsTable.css';

export interface SignalsTableProps {
    className?: string;

    data?: SignalFormattedSchema[];
    isFullyLoaded?: boolean;
    isLoading?: boolean;
    fetchNextPage?: () => void;

    isShortView?: boolean;
    isOneCar?: boolean;

    onClick: (signal: SignalFormattedSchema) => void;
}

export const SignalsTable: React.FC<SignalsTableProps> = React.memo(function SignalsTable({
    className,
    onClick,
    data,
    isFullyLoaded,
    isLoading,
    isShortView,
    isOneCar,
    fetchNextPage,
}) {
    const TableHead = React.memo(() => {
        if (isShortView) {
            return (
                <>
                    <Table.Th>{i18n('Date and time')}</Table.Th>
                    <Table.Th>{i18n('Signal')}</Table.Th>
                    <Table.Th>{i18n('Ride')}</Table.Th>
                </>
            );
        }

        return (
            <>
                <Table.Th>{i18n('Date and time')}</Table.Th>
                <Table.Th>{i18n('Level')}</Table.Th>
                <Table.Th>{i18n('Signal')}</Table.Th>
                {!isOneCar && <Table.Th>{i18n('Car')}</Table.Th>}
                <Table.Th>{i18n('Ride')}</Table.Th>
                <Table.Th>{i18n('Duration, min')}</Table.Th>
            </>
        );
    });

    return (
        <div className={cn(styles.container, className)}>
            {data && Boolean(data.length) && (
                <Table>
                    <Table.Head>
                        <Table.Row>
                            <Table.Th gap />
                            <TableHead />
                            <Table.Th gap />
                        </Table.Row>
                    </Table.Head>
                    <Table.Body>
                        {data.map((signal) => {
                            return (
                                <SignalsTableRow
                                    key={signal.tag_id || signal.event_id}
                                    signal={signal}
                                    onClick={onClick}
                                    isShortView={isShortView}
                                    isOneCar={isOneCar}
                                />
                            );
                        })}
                    </Table.Body>
                </Table>
            )}

            {isFullyLoaded && !(data && data.length) && (
                <div className={styles.empty}>{i18n('No signals to display')}</div>
            )}

            {isLoading ? (
                <div className={styles.spin}>
                    <Spin />
                </div>
            ) : (
                !isFullyLoaded &&
                fetchNextPage && (
                    <Link
                        className={styles.more}
                        onClick={fetchNextPage}
                    >
                        {i18n('Show more')}
                    </Link>
                )
            )}
        </div>
    );
});
