import * as React from 'react';
import { useHistory, useLocation } from 'react-router-dom';
import cn from 'classnames/bind';

import { EMPTY_DATA, MODAL_OBJECT_ID_CGI, MODAL_OBJECT_TYPE_CGI } from 'constants/constants';

import { durationUTCBetween } from 'utils/date/durationUTCBetween';
import shortDateTime from 'utils/date/shortDateTime';

import { formatCarModel, getCarNumber } from 'entities/Car';
import { SignalPriority } from 'entities/Signal/consts/SignalPriority';
import { getSignalTypeTitle } from 'entities/Signal/helpers/getSignalTypeTitle/getSignalTypeTitle';
import { SignalFormattedSchema } from 'entities/Signal/types/SignalFormattedSchema';
import { SignalMediaDetailsSchema } from 'entities/Signal/types/SignalMediaDetailsSchema';

import { Path } from 'shared/consts/Path';
import { generateRouterPath } from 'shared/helpers/generateRouterPath/generateRouterPath';
import { Link } from 'shared/ui/Link/Link';
import { Table } from 'shared/ui/Table/Table';
import { TextWithDot } from 'shared/ui/TextWithDot/TextWithDot';

import { SessionTime } from 'components/Cars/CarCard/CarSessionsTable/sessionTime';
import { SCORING_TRACE_TAG } from 'components/Cars/constants';
import { ModalObjectTypes } from 'components/types';
import { DurationUTCBetween } from 'components/ui/DurationUTCBetween';

import { i18n } from 'features/SignalsTable/ui/SignalsTableRow/SignalsTableRow.i18n';

import PlayIcon from 'shared/ui/Icons/images/play-filled-16.inline.svg';

import styles from 'features/SignalsTable/ui/SignalsTableRow/SignalsTableRow.css';

export interface SignalsTableRowProps {
    className?: string;

    isShortView?: boolean;
    isOneCar?: boolean;

    signal: SignalFormattedSchema;
    onClick: (signal: SignalFormattedSchema) => void;
}

const cx = cn.bind(styles);

const SIGNAL_PRIORITY: Record<SignalPriority, string> = {
    [SignalPriority.CRITICAL]: i18n('Critical'),
    [SignalPriority.NORMAL]: i18n('Normal'),
    [SignalPriority.WARNING]: i18n('Normal'),
};

export const SignalsTableRow: React.FC<SignalsTableRowProps> = function SignalsTableRow({
    className,
    isShortView,
    isOneCar,
    signal,
    onClick,
}) {
    const location = useLocation();
    const history = useHistory();

    const [imageError, setImageError] = React.useState<boolean>();
    const { name, since, until, description, car, is_actual, details, trace } = signal;
    const { object_id: traceId } = trace || {};
    const { priority, display_name, type, is_permanent } = description;
    const time = since ? shortDateTime(since) : EMPTY_DATA;

    const Level = React.memo(() => {
        return (
            <>
                <div className={styles.priority}>{SIGNAL_PRIORITY[priority]}</div>
                {is_actual && <div className={styles.actual}>{i18n('Actual')}</div>}
            </>
        );
    });

    const Car = React.memo(() => {
        const { details, object_id } = car || {};
        const { number, model } = details || {};

        return details && object_id ? (
            <>
                <Link
                    href={generateRouterPath(Path.CAR, { id: object_id })}
                    className={styles.carNumber}
                >
                    {getCarNumber(number)}
                </Link>
                <div className={styles.carModel}>{formatCarModel(model)}</div>
            </>
        ) : null;
    });

    const Duration = React.memo(() => {
        if (is_permanent) {
            return null;
        }

        return name !== SCORING_TRACE_TAG ? <DurationUTCBetween duration={durationUTCBetween(since, until)} /> : null;
    });

    const onImageError = React.useCallback(() => {
        setImageError(true);
    }, []);

    const Signal = React.memo(() => {
        const media: SignalMediaDetailsSchema = details && details.media;
        const signalTypeTitle = getSignalTypeTitle(type);

        return (
            <div className={styles.signal}>
                {media?.photo ? (
                    <div
                        className={styles.signalPhoto}
                        onClick={onRowClick}
                    >
                        {media?.video && <PlayIcon className={styles.playIcon} />}

                        {!imageError && (
                            <img
                                onError={onImageError}
                                src={media.photo.presigned_url.link}
                            />
                        )}
                    </div>
                ) : null}
                <div className={styles.signalDetails}>
                    <Link
                        onClick={onRowClick}
                        className={cx(styles.signalName, { bold: isShortView })}
                    >
                        {display_name || name}
                    </Link>
                    {(signalTypeTitle || isShortView) && (
                        <div className={styles.signalType}>
                            {isShortView && (
                                <div className={cx(styles.priority, { red: priority === SignalPriority.CRITICAL })}>
                                    {priority}
                                </div>
                            )}
                            {signalTypeTitle}
                        </div>
                    )}
                </div>
            </div>
        );
    });

    const onTraceClick = React.useCallback(() => {
        let searchParams = new URLSearchParams(location.search);
        searchParams.set(MODAL_OBJECT_TYPE_CGI, ModalObjectTypes.SESSION);
        searchParams.set(MODAL_OBJECT_ID_CGI, traceId ?? '');
        history.push(`${location.pathname}?${searchParams}`);
    }, [history, location.pathname, location.search, traceId]);

    const Ride = React.memo(() => {
        return trace ? (
            <Link
                onClick={onTraceClick}
                className={styles.ride}
            >
                <SessionTime
                    key={trace.object_id}
                    start={trace.details?.since}
                    finish={trace.details?.until}
                />
            </Link>
        ) : null;
    });

    const onRowClick = React.useCallback(() => {
        onClick(signal);
    }, [onClick, signal]);

    if (isShortView) {
        return (
            <Table.Row className={styles.row}>
                <Table.Td gap />
                <Table.Td className={styles.cell}>{time}</Table.Td>
                <Table.Td className={styles.cell}>
                    <Signal />
                </Table.Td>
                <Table.Td className={styles.cell}>
                    <Ride />
                </Table.Td>
                <Table.Td gap />
            </Table.Row>
        );
    }

    return (
        <Table.Row className={styles.row}>
            <Table.Td gap />
            <Table.Td className={styles.cell}>
                <TextWithDot
                    className={cx(styles.time, styles[priority])}
                    pulsar={is_actual && priority === SignalPriority.CRITICAL}
                >
                    {time}
                </TextWithDot>
            </Table.Td>
            <Table.Td className={styles.cell}>
                <Level />
            </Table.Td>
            <Table.Td className={styles.cell}>
                <Signal />
            </Table.Td>
            {!isOneCar && (
                <Table.Td className={styles.cell}>
                    <Car />
                </Table.Td>
            )}
            <Table.Td className={styles.cell}>
                <Ride />
            </Table.Td>
            <Table.Td className={styles.cell}>
                <Duration />
            </Table.Td>
            <Table.Td gap />
        </Table.Row>
    );
};
