import * as React from 'react';
import { useCallback, useEffect, useState } from 'react';

import { UserDetailsForm } from 'features/UserDetails/ui/UserDetailsForm/UserDetailsForm';

import { fetchUsersSearch } from 'entities/User/api/fetchUsersSearch/fetchUsersSearch';
import { postUserCreate } from 'entities/User/api/postUserCreate/postUserCreate';
import { useUserPossibleRoles } from 'entities/User/api/useUserPossibleRoles/useUserPossibleRoles';
import { UserDetailsFormSchema } from 'entities/User/types/UserDetailsFormSchema';

import { ButtonColor } from 'shared/consts/ButtonColor';
import { ButtonSize } from 'shared/consts/ButtonSize';
import { getFetchErrorMessage } from 'shared/helpers/getFetchErrorMessage/getFetchErrorMessage';
import { useCacheRequestContext } from 'shared/hooks/useCacheRequestContext/useCacheRequestContext';
import { useEqualDataForm } from 'shared/hooks/useEqualDataForm/useEqualDataForm';
import { useFormController } from 'shared/hooks/useFormController/useFormController';
import { Button } from 'shared/ui/Button/Button';
import { MenuItemOptions } from 'shared/ui/MenuItem/MenuItem';
import { ModalContainer } from 'shared/ui/ModalContainer/ModalContainer';

import formatPhoneValue from 'components/GlobalSidebar/ModalUserView/UserForm/formatPhoneValue';

import { i18n } from 'features/UserDetails/ui/UserCreateModal/UserCreateModal.i18n';

export interface UserCreateModalProps {
    onClose: () => void;
    onUserCreate?: () => void;
}

const DELETED_STATUS = 'deleted';
const initForm: UserDetailsFormSchema = {
    first_name: '',
    last_name: '',
    address: '',
    phone: '',
    email: '',
    role: '',
};

export const UserCreateModal: React.FC<UserCreateModalProps> = function UserCreateModal({ onClose, onUserCreate }) {
    const { error: possibleRolesError, data: possibleRoles } = useUserPossibleRoles();
    const cache = useCacheRequestContext();

    const [rolesOptions, setRolesOptions] = useState<MenuItemOptions[]>([]);

    const { getValues, validate, setError, controller } = useFormController<OptionalRecord<UserDetailsFormSchema>>();
    const { isEqualData, onFormChangeHandler } = useEqualDataForm<UserDetailsFormSchema>(initForm);

    useEffect(() => {
        if (possibleRolesError) {
            setError('_serverError', getFetchErrorMessage(possibleRolesError));
        } else {
            const rolesOptions =
                possibleRoles?.map((role) => {
                    return {
                        value: role,
                        label: role,
                    };
                }) ?? [];

            setRolesOptions(rolesOptions);
        }
    }, [setError, setRolesOptions, possibleRoles, possibleRolesError]);

    const checkIfUserExist = useCallback(
        (phone: string): Promise<boolean> => {
            const searchPhoneValue = formatPhoneValue(phone);

            return fetchUsersSearch(searchPhoneValue, cache).then((response) => {
                return Boolean(response?.filter((user) => user.status !== DELETED_STATUS)?.length);
            });
        },
        [cache],
    );

    const onSubmitHandler = React.useCallback(async () => {
        if (validate()) {
            const formData = getValues();

            if (formData?.phone) {
                try {
                    const isUserExist = await checkIfUserExist(formData.phone);

                    if (!isUserExist) {
                        await postUserCreate(formData, cache);
                        onUserCreate?.();
                        onClose();
                    } else {
                        setError('phone', [
                            {
                                message: i18n('User with this phone number already exists'),
                                code: '',
                            },
                        ]);
                    }
                } catch (error) {
                    setError('_serverError', getFetchErrorMessage(error));
                }
            }
        }
    }, [getValues, validate, checkIfUserExist, onClose, cache, setError, onUserCreate]);

    return (
        <>
            <ModalContainer
                title={i18n('Add a user')}
                hasClose
                controls={
                    <Button
                        color={ButtonColor.PRIMARY}
                        size={ButtonSize.M}
                        label={i18n('Add user')}
                        disabled={isEqualData}
                        onClick={onSubmitHandler}
                    />
                }
                onClose={onClose}
            >
                <UserDetailsForm
                    initial={initForm}
                    controller={controller}
                    onFormChange={onFormChangeHandler}
                    roleInputValues={rolesOptions}
                />
            </ModalContainer>
        </>
    );
};
