import * as React from 'react';
import cn from 'classnames/bind';

import { EMPTY_DATA } from 'constants/constants';

import { enableCarAssignmentFlag } from 'utils/enableCarAssignmentFlag';
import { isUserDetailsAddressFlag } from 'utils/isUserDetailsAddressFlag';

import { PostUserStatus, postUserStatus } from 'features/UserDetails/api/postUserStatus/postUserStatus';
import { UserBlockDialog } from 'features/UserDetails/ui/UserBlockDialog/UserBlockDialog';
import { UserEditModal } from 'features/UserDetails/ui/UserEditModal/UserEditModal';

import { formatCarModel, formatCarNumber } from 'entities/Car';
import { getUserRoles, UserStatusInfo } from 'entities/User';
import { UseUserResource } from 'entities/User/api/useUser/useUser';
import { UseUserMemberResource } from 'entities/User/api/useUserMember/useUserMember';
import { UsePerformedTagsResource } from 'entities/User/api/useUserPerformedTags/useUserPerformedTags';
import { UserStatus } from 'entities/User/consts/UserStatus';
import { getUserCarTag } from 'entities/User/helpers/getUserCarTag/getUserCarTag';
import { getUserSetupEmailAddress } from 'entities/User/helpers/getUserSetupEmailAddress/getUserSetupEmailAddress';
import { getUserSetupPhoneNumber } from 'entities/User/helpers/getUserSetupPhoneNumber/getUserSetupPhoneNumber';

import { ButtonColor } from 'shared/consts/ButtonColor';
import { ButtonSize } from 'shared/consts/ButtonSize';
import { HttpStatusCode } from 'shared/consts/HttpStatusCode';
import { Path } from 'shared/consts/Path';
import { FetchError } from 'shared/helpers/fetchRequest/fetchRequest';
import { formatPhoneNumber } from 'shared/helpers/formatPhoneNumber/formatPhoneNumber';
import { generateRouterPath } from 'shared/helpers/generateRouterPath/generateRouterPath';
import { Button } from 'shared/ui/Button/Button';
import { DetailsContainer, DetailsContainerItemOptions } from 'shared/ui/DetailsContainer/DetailsContainer';
import { FormContainer } from 'shared/ui/FormContainer/FormContainer';
import { Link } from 'shared/ui/Link/Link';
import { Widget } from 'shared/ui/Widget/Widget';

import { i18n } from 'features/UserDetails/ui/UserDetails/UserDetails.i18n';

import BanIcon from 'shared/ui/Icons/images/ban-16.inline.svg';
import EditIcon from 'shared/ui/Icons/images/edit-outline-16.inline.svg';

import styles from 'features/UserDetails/ui/UserDetails/UserDetails.css';

export interface UserDetailsProps {
    className?: string;
    userResource: UseUserResource;
    memberResource: UseUserMemberResource;
    performedTagsResource: UsePerformedTagsResource;
}

const cx = cn.bind(styles);

export const UserDetails: React.FC<UserDetailsProps> = function UserDetails({
    className,
    userResource,
    memberResource,
    performedTagsResource,
}) {
    const [isEditModalOpen, setEditModalOpen] = React.useState<boolean>(false);
    const [isBlockDialogOpen, setBlockDialogOpen] = React.useState<boolean>(false);

    const user = userResource.read();
    const member = memberResource.read();
    const performedTags = performedTagsResource.read();

    const { id, status, address, setup } = user;
    const { roles } = member;

    const performedCarTag = React.useMemo(() => {
        return getUserCarTag(performedTags);
    }, [performedTags]);

    const onEditModalOpen = React.useCallback(() => {
        setEditModalOpen(true);
    }, []);

    const onEditModalClose = React.useCallback(() => {
        setEditModalOpen(false);
    }, []);

    const onEditModalSubmitHandler = React.useCallback(() => {
        memberResource.reload();
    }, [memberResource]);

    const onBlockDialogOpen = React.useCallback(() => {
        setBlockDialogOpen(true);
    }, []);

    const onBlockDialogClose = React.useCallback(() => {
        setBlockDialogOpen(false);
    }, []);

    const onStatusChange = React.useCallback(
        async (status: PostUserStatus) => {
            try {
                await postUserStatus(id, status);
            } catch (error) {
                if (error instanceof FetchError) {
                    if (error.status === HttpStatusCode.BAD_REQUEST) {
                        console.error(error.res.error_details?.ui_message);

                        return;
                    }
                }
            }

            userResource.reload();
            memberResource.reload();
        },
        [userResource, memberResource, id],
    );

    const onVerifySubmitHandler = React.useCallback(async () => {
        await onStatusChange(UserStatus.ACTIVE);

        // @todo: add notification
    }, [onStatusChange]);

    const onRejectSubmitHandler = React.useCallback(async () => {
        await onStatusChange(UserStatus.REJECTED);

        // @todo: add notification
    }, [onStatusChange]);

    const onBlockSubmitHandler = React.useCallback(async () => {
        await onStatusChange(status === UserStatus.ACTIVE ? UserStatus.BLOCKED : UserStatus.ACTIVE);

        onBlockDialogClose();

        // @todo: add notification
    }, [status, onStatusChange, onBlockDialogClose]);

    const detailsItems = React.useMemo(() => {
        const performedCar = performedCarTag?.object_info;

        return [
            {
                label: i18n('Role'),
                id: 'role',
                value: getUserRoles(roles).join(', ') || EMPTY_DATA,
            },

            {
                label: i18n('Status'),
                id: 'status',
                value: <UserStatusInfo status={status} />,
            },

            {
                label: i18n('E-mail'),
                id: 'email',
                value: getUserSetupEmailAddress(setup),
            },

            {
                label: i18n('Phone'),
                id: 'phone',
                value: formatPhoneNumber(getUserSetupPhoneNumber(setup)),
            },

            performedCar && enableCarAssignmentFlag()
                ? {
                      label: i18n('Assigned car'),
                      id: 'car',
                      value: (
                          <Link href={generateRouterPath(Path.CAR, { id: performedCar.id })}>
                              {`${formatCarModel(performedCar.model_id)} ${formatCarNumber(performedCar.number)}`}
                          </Link>
                      ),
                  }
                : {},

            isUserDetailsAddressFlag()
                ? {
                      label: i18n('Address'),
                      id: 'address',
                      value: address,
                  }
                : {},
        ] as DetailsContainerItemOptions[];
    }, [roles, status, address, setup, performedCarTag]);

    const isActive = status === UserStatus.ACTIVE;
    const isBlocked = status === UserStatus.BLOCKED;

    return (
        <>
            <Widget className={cn(styles.content, className)}>
                <FormContainer>
                    <h3 className={styles.title}>{i18n('User details')}</h3>

                    <DetailsContainer items={detailsItems} />

                    <div className={styles.controls}>
                        {(status === UserStatus.SCREENING || status === UserStatus.REJECTED) && (
                            <Button
                                color={ButtonColor.SECONDARY}
                                size={ButtonSize.M}
                                label={i18n('Verify')}
                                onClick={onVerifySubmitHandler}
                            />
                        )}

                        <Button
                            color={isActive ? ButtonColor.PRIMARY : ButtonColor.SECONDARY}
                            size={ButtonSize.M}
                            icon={EditIcon}
                            label={i18n('Edit')}
                            onClick={onEditModalOpen}
                        />

                        {status === UserStatus.SCREENING && (
                            <Button
                                color={ButtonColor.ALARM}
                                size={ButtonSize.M}
                                icon={BanIcon}
                                label={i18n('Reject')}
                                onClick={onRejectSubmitHandler}
                            />
                        )}

                        {(isActive || isBlocked) && (
                            <Button
                                className={cx(styles.blockButton, { first: isBlocked })}
                                color={isBlocked ? ButtonColor.PRIMARY : ButtonColor.ALARM}
                                size={ButtonSize.M}
                                icon={BanIcon}
                                label={isBlocked ? i18n('Unblock user') : i18n('Block user')}
                                onClick={onBlockDialogOpen}
                            />
                        )}
                    </div>
                </FormContainer>
            </Widget>

            {isEditModalOpen && (
                <UserEditModal
                    userResource={userResource}
                    performedTagsResource={performedTagsResource}
                    onClose={onEditModalClose}
                    onModalSubmit={onEditModalSubmitHandler}
                />
            )}

            {isBlockDialogOpen && (
                <UserBlockDialog
                    unblock={isBlocked}
                    onClose={onBlockDialogClose}
                    onDialogSubmit={onBlockSubmitHandler}
                />
            )}
        </>
    );
};
