import * as React from 'react';

import { enableCarAssignmentFlag } from 'utils/enableCarAssignmentFlag';
import { isUserDetailsAddressFlag } from 'utils/isUserDetailsAddressFlag';

import { CarInput } from 'entities/Car';
import { UserDetailsFormSchema } from 'entities/User/types/UserDetailsFormSchema';

import { InputSize } from 'shared/consts/InputSize';
import { validateEmail } from 'shared/helpers/validateEmail/validateEmail';
import { validatePhoneNumber } from 'shared/helpers/validatePhoneNumber/validatePhoneNumber';
import { validateRequired } from 'shared/helpers/validateRequired/validateRequired';
import { useForm, UseFormValidation } from 'shared/hooks/useForm/useForm';
import { UseFormControllerSub } from 'shared/hooks/useFormController/useFormController';
import { ErrorLabel } from 'shared/ui/ErrorLabel/ErrorLabel';
import { ErrorMessage } from 'shared/ui/ErrorMessage/ErrorMessage';
import { FormContainer } from 'shared/ui/FormContainer/FormContainer';
import { FormErrorContainer } from 'shared/ui/FormErrorContainer/FormErrorContainer';
import { FormLabel } from 'shared/ui/FormLabel/FormLabel';
import { Input } from 'shared/ui/Input/Input';
import { InputEmail } from 'shared/ui/InputEmail/InputEmail';
import { InputPhoneNumber } from 'shared/ui/InputPhoneNumber/InputPhoneNumber';
import { MenuItemOptions } from 'shared/ui/MenuItem/MenuItem';
import { Select } from 'shared/ui/Select/Select';

import { i18n } from 'features/UserDetails/ui/UserDetailsForm/UserDetailsForm.i18n';

import styles from 'features/UserDetails/ui/UserDetailsForm/UserDetailsForm.css';

export interface UserDetailsFormProps {
    className?: string;
    initial?: UserDetailsFormSchema;
    controller?: UseFormControllerSub<UserDetailsFormSchema>;
    showCarInput?: boolean;
    roleInputValues?: Optional<MenuItemOptions[]>;

    onFormChange?(state: OptionalRecord<UserDetailsFormSchema>): void;
}

const INIT_FORM: OptionalRecord<UserDetailsFormSchema> = {
    first_name: '',
    last_name: '',
    phone: '',
    email: '',
    address: '',
};

const VALIDATION_RULES: UseFormValidation<OptionalRecord<UserDetailsFormSchema>> = {
    first_name: validateRequired(i18n('Enter name')),
    last_name: validateRequired(i18n('Enter surname')),
    phone: validatePhoneNumber(),
    email: validateEmail(),
};

export const UserDetailsForm: React.FC<UserDetailsFormProps> = function UserDetailsForm({
    className,
    initial,
    controller,
    onFormChange,
    showCarInput,
    roleInputValues,
}) {
    const { update, watch, errors } = useForm<OptionalRecord<UserDetailsFormSchema>>({
        init: { ...INIT_FORM, ...initial },
        validation: VALIDATION_RULES,
        controller,
        onFormChange,
    });

    return (
        <FormContainer className={className}>
            <FormLabel title={i18n('Personal data')}>
                <div className={styles.item}>
                    <FormErrorContainer error={<ErrorLabel errors={errors.first_name} />}>
                        <Input
                            inputSize={InputSize.M}
                            placeholder={i18n('Name')}
                            value={watch('first_name')}
                            hasError={Boolean(errors.first_name)}
                            hasClear
                            onInputChange={update('first_name')}
                        />
                    </FormErrorContainer>

                    <FormErrorContainer error={<ErrorLabel errors={errors.last_name} />}>
                        <Input
                            inputSize={InputSize.M}
                            placeholder={i18n('Surname')}
                            value={watch('last_name')}
                            hasError={Boolean(errors.last_name)}
                            hasClear
                            onInputChange={update('last_name')}
                        />
                    </FormErrorContainer>
                </div>
            </FormLabel>

            <FormLabel title={i18n('Contact information')}>
                <div className={styles.item}>
                    <FormErrorContainer error={<ErrorLabel errors={errors.phone} />}>
                        <InputPhoneNumber
                            inputMode="numeric"
                            inputSize={InputSize.M}
                            placeholder={i18n('Phone')}
                            value={watch('phone')}
                            hasError={Boolean(errors.phone)}
                            hasClear
                            onInputChange={update('phone')}
                        />
                    </FormErrorContainer>

                    <FormErrorContainer error={<ErrorLabel errors={errors.email} />}>
                        <InputEmail
                            inputSize={InputSize.M}
                            placeholder={i18n('E-mail')}
                            value={watch('email')}
                            hasError={Boolean(errors.email)}
                            hasClear
                            onInputChange={update('email')}
                        />
                    </FormErrorContainer>

                    {isUserDetailsAddressFlag() && (
                        <Input
                            inputSize={InputSize.M}
                            placeholder={i18n('Address')}
                            value={watch('address')}
                            hasClear
                            onInputChange={update('address')}
                        />
                    )}
                </div>
            </FormLabel>

            {showCarInput && enableCarAssignmentFlag() && (
                <FormLabel title={i18n('Car')}>
                    <div className={styles.item}>
                        <FormErrorContainer error={<ErrorLabel errors={errors.car} />}>
                            <CarInput
                                inputSize={InputSize.M}
                                placeholder={i18n('Car model or licence plate number')}
                                hasError={Boolean(errors.car)}
                                initialValue={watch('car')}
                                onCarInputChange={update('car')}
                            />
                        </FormErrorContainer>
                    </div>
                </FormLabel>
            )}

            {roleInputValues && (
                <FormLabel title={i18n('Role')}>
                    <div className={styles.item}>
                        <FormErrorContainer error={<ErrorLabel errors={errors.role} />}>
                            <Select
                                placeholder={i18n('Role')}
                                items={roleInputValues}
                                hasError={Boolean(errors.role)}
                                checkedMenu={roleInputValues[0]?.value ?? ''}
                                onSelectChange={update('role')}
                                disabled={!roleInputValues.length}
                            />
                        </FormErrorContainer>
                    </div>
                </FormLabel>
            )}

            <ErrorMessage error={errors._serverError} />
        </FormContainer>
    );
};
