import * as React from 'react';
import { useHistory } from 'react-router-dom';

import { deleteUser } from 'features/UserDetails/api/deleteUser/deleteUser';
import { postUserEdit } from 'features/UserDetails/api/postUserEdit/postUserEdit';
import { UserDeleteDialog } from 'features/UserDetails/ui/UserDeleteDialog/UserDeleteDialog';
import { UserDetailsForm } from 'features/UserDetails/ui/UserDetailsForm/UserDetailsForm';

import { formatCarModel } from 'entities/Car';
import { UseUserResource } from 'entities/User/api/useUser/useUser';
import { UsePerformedTagsResource } from 'entities/User/api/useUserPerformedTags/useUserPerformedTags';
import { getUserCarTag } from 'entities/User/helpers/getUserCarTag/getUserCarTag';
import { getUserSetupEmailAddress } from 'entities/User/helpers/getUserSetupEmailAddress/getUserSetupEmailAddress';
import { getUserSetupPhoneNumber } from 'entities/User/helpers/getUserSetupPhoneNumber/getUserSetupPhoneNumber';
import { UserDetailsFormSchema } from 'entities/User/types/UserDetailsFormSchema';

import { ButtonColor } from 'shared/consts/ButtonColor';
import { ButtonSize } from 'shared/consts/ButtonSize';
import { Path } from 'shared/consts/Path';
import { getFetchErrorMessage } from 'shared/helpers/getFetchErrorMessage/getFetchErrorMessage';
import { useEqualDataForm } from 'shared/hooks/useEqualDataForm/useEqualDataForm';
import { useFormController } from 'shared/hooks/useFormController/useFormController';
import { Button } from 'shared/ui/Button/Button';
import { ModalContainer, ModalContainerProps } from 'shared/ui/ModalContainer/ModalContainer';

import { i18n } from 'features/UserDetails/ui/UserEditModal/UserEditModal.i18n';

import TrashIcon from 'shared/ui/Icons/images/trash-outline-16.inline.svg';

export interface UserEditModalProps extends Pick<ModalContainerProps, 'onClose'> {
    userResource: UseUserResource;
    performedTagsResource: UsePerformedTagsResource;

    onModalSubmit?(): void;
}

export const UserEditModal: React.FC<UserEditModalProps> = function UserEditModal({
    userResource,
    performedTagsResource,
    onClose,
    onModalSubmit,
}) {
    const history = useHistory();

    const [isDeleteDialogOpen, setDeleteDialogOpen] = React.useState<boolean>(false);

    const { getValues, validate, setError, controller } = useFormController<OptionalRecord<UserDetailsFormSchema>>();

    const user = userResource.read();
    const performedTags = performedTagsResource.read();

    const { id, first_name: firstName, last_name: lastName, address, setup } = user;

    const performedCarTag = getUserCarTag(performedTags);

    const initForm = React.useMemo(
        (): UserDetailsFormSchema => ({
            first_name: firstName,
            last_name: lastName,
            address,
            phone: getUserSetupPhoneNumber(setup),
            email: getUserSetupEmailAddress(setup),
            car: performedCarTag && {
                model_name: formatCarModel(performedCarTag.object_info?.model_id) ?? '',
                ...performedCarTag.object_info,
            },
        }),

        [firstName, lastName, address, setup, performedCarTag],
    );

    const onDeleteDialogOpen = React.useCallback(() => {
        setDeleteDialogOpen(true);
    }, []);

    const onDeleteDialogClose = React.useCallback(() => {
        setDeleteDialogOpen(false);
    }, []);

    const onSubmitHandler = React.useCallback(async () => {
        if (validate()) {
            const formData = getValues();

            try {
                await postUserEdit(id, formData as UserDetailsFormSchema, performedCarTag);

                // @todo: add notification
            } catch (error) {
                setError('_serverError', getFetchErrorMessage(error));
            }

            userResource.reload();
            performedTagsResource.reload();

            if (onClose) {
                onClose();
            }

            if (onModalSubmit) {
                onModalSubmit();
            }
        }
    }, [
        onModalSubmit,
        onClose,
        getValues,
        validate,
        setError,
        userResource,
        performedTagsResource,
        id,
        performedCarTag,
    ]);

    const onDeleteSubmitHandler = React.useCallback(async () => {
        try {
            await deleteUser(id);

            // @todo: add notification
        } catch (error) {
            return setError('_serverError', getFetchErrorMessage(error));
        }

        history.push(Path.USERS);
    }, [history, setError, id]);

    const { isEqualData, onFormChangeHandler } = useEqualDataForm<UserDetailsFormSchema>(initForm);

    return (
        <>
            <ModalContainer
                title={i18n('Edit profile')}
                hasClose
                controls={
                    <>
                        <Button
                            color={ButtonColor.PRIMARY}
                            size={ButtonSize.M}
                            label={i18n('Save changes')}
                            disabled={isEqualData}
                            onClick={onSubmitHandler}
                        />

                        <Button
                            color={ButtonColor.ALARM}
                            size={ButtonSize.M}
                            icon={TrashIcon}
                            label={i18n('Delete user')}
                            onClick={onDeleteDialogOpen}
                        />
                    </>
                }
                onClose={onClose}
            >
                <UserDetailsForm
                    initial={initForm}
                    controller={controller}
                    onFormChange={onFormChangeHandler}
                    showCarInput
                />
            </ModalContainer>

            {isDeleteDialogOpen && (
                <UserDeleteDialog
                    onDialogSubmit={onDeleteSubmitHandler}
                    onClose={onDeleteDialogClose}
                />
            )}
        </>
    );
};
