import * as React from 'react';

import { getI18nLocale } from 'utils/language/getI18nLocale';

import { USER_DOCUMENTS_DATE_FORMAT } from 'features/UserDocuments/consts/UserDocumentsDateFormat';
import { UserDocumentDetailsItem } from 'features/UserDocuments/ui/UserDocumentDetails/UserDocumentDetails';
import { UserDocumentsItem } from 'features/UserDocuments/ui/UserDocumentsItem/UserDocumentsItem';
import { UserEditDocumentDriverLicenseModal } from 'features/UserDocuments/ui/UserEditDocumentDriverLicenseModal/UserEditDocumentDriverLicenseModal';

import { getUserDocumentDriverLicenseField, UseUserDocumentPhotoRes, UseUserResource } from 'entities/User';

import { getDateString } from 'shared/helpers/getDateString/getDateString';

import { i18n } from 'features/UserDocuments/ui/UserDocumentDriverLicense/UserDocumentDriverLicense.i18n';

export interface UserDocumentDriverLicenseProps {
    className?: string;
    resource: UseUserResource;
}

const EMPTY_PHOTOS: UseUserDocumentPhotoRes = { lf: {}, lb: {} };

// @todo: add storybook test
export const UserDocumentDriverLicense: React.FC<UserDocumentDriverLicenseProps> = function UserDocumentDriverLicense({
    className,
    resource,
}) {
    const [isEditModalOpen, setEditModalOpen] = React.useState<boolean>(false);

    const documentsResource = resource.read();

    const {
        photos: { driving_license: drivingLicensePhotos, submitted_at: submittedAt },
        documents: { driving_license: drivingLicenseDetails },
        driving_license_revision: updatedAt,
    } = documentsResource;

    const onEditModalOpen = React.useCallback(() => {
        setEditModalOpen(true);
    }, []);

    const onEditModalClose = React.useCallback(() => {
        setEditModalOpen(false);
    }, []);

    const locale = getI18nLocale();

    const hasData = React.useMemo(
        () => drivingLicensePhotos && Boolean(Object.keys(drivingLicensePhotos).length),
        [drivingLicensePhotos],
    );

    const photos = React.useMemo(() => ({ ...EMPTY_PHOTOS, ...drivingLicensePhotos }), [drivingLicensePhotos]);
    const details = React.useMemo((): UserDocumentDetailsItem[] => {
        if (!drivingLicenseDetails) {
            return [];
        }

        return [
            {
                label: getUserDocumentDriverLicenseField('first_name'),
                value: drivingLicenseDetails.first_name,
            },

            {
                label: getUserDocumentDriverLicenseField('last_name'),
                value: drivingLicenseDetails.last_name,
            },

            {
                label: getUserDocumentDriverLicenseField('front_country'),
                value: drivingLicenseDetails.front_country,
            },

            {
                label: getUserDocumentDriverLicenseField('birth_date'),
                value: getDateString(drivingLicenseDetails.birth_date, locale, USER_DOCUMENTS_DATE_FORMAT),
            },

            {
                label: getUserDocumentDriverLicenseField('categories'),
                value: Array.isArray(drivingLicenseDetails.categories)
                    ? drivingLicenseDetails.categories.join(', ').toUpperCase()
                    : drivingLicenseDetails.categories?.toUpperCase(),
            },

            {
                label: getUserDocumentDriverLicenseField('issue_date'),
                value: getDateString(drivingLicenseDetails.issue_date, locale, USER_DOCUMENTS_DATE_FORMAT),
            },

            {
                label: getUserDocumentDriverLicenseField('categories_b_valid_from_date'),
                value: getDateString(
                    drivingLicenseDetails.categories_b_valid_from_date,
                    locale,
                    USER_DOCUMENTS_DATE_FORMAT,
                ),
            },

            {
                label: getUserDocumentDriverLicenseField('categories_b_valid_to_date'),
                value: getDateString(
                    drivingLicenseDetails.categories_b_valid_to_date,
                    locale,
                    USER_DOCUMENTS_DATE_FORMAT,
                ),
            },

            {
                label: getUserDocumentDriverLicenseField('number_front'),
                value: drivingLicenseDetails.number_front,
            },

            {
                label: getUserDocumentDriverLicenseField('number_back'),
                value: drivingLicenseDetails.number_back,
            },
        ];
    }, [locale, drivingLicenseDetails]);

    return (
        <>
            <UserDocumentsItem
                className={className}
                title={i18n('Driver license')}
                submittedAt={submittedAt}
                updatedAt={updatedAt}
                photos={photos}
                details={details}
                onEdit={hasData ? onEditModalOpen : undefined}
            />

            {isEditModalOpen && (
                <UserEditDocumentDriverLicenseModal
                    resource={resource}
                    onClose={onEditModalClose}
                />
            )}
        </>
    );
};
