import * as React from 'react';

import { getI18nLocale } from 'utils/language/getI18nLocale';

import { USER_DOCUMENTS_DATE_FORMAT } from 'features/UserDocuments/consts/UserDocumentsDateFormat';
import { UserDocumentDetailsItem } from 'features/UserDocuments/ui/UserDocumentDetails/UserDocumentDetails';
import { UserDocumentsItem } from 'features/UserDocuments/ui/UserDocumentsItem/UserDocumentsItem';
import { UserEditDocumentPassportModal } from 'features/UserDocuments/ui/UserEditDocumentPassportModal/UserEditDocumentPassportModal';

import {
    getUserDocumentPassportField,
    getUserDocumentPassportGender,
    UseUserDocumentPhotoRes,
    UseUserResource,
} from 'entities/User';

import { getDateString } from 'shared/helpers/getDateString/getDateString';

import { i18n } from 'features/UserDocuments/ui/UserDocumentPassport/UserDocumentPassport.i18n';

export interface UserDocumentPassportProps {
    className?: string;
    resource: UseUserResource;
}

const EMPTY_PHOTOS: UseUserDocumentPhotoRes = { pb: {}, ps: {} };

// @todo: add storybook test
export const UserDocumentPassport: React.FC<UserDocumentPassportProps> = function UserDocumentPassport({
    className,
    resource,
}) {
    const [isEditModalOpen, setEditModalOpen] = React.useState<boolean>(false);

    const documentsResource = resource.read();

    const {
        photos: { passport: passportPhotos, submitted_at: submittedAt },
        documents: { passport: passportDetails },
        passport_revision: updatedAt,
    } = documentsResource;

    const onEditModalOpen = React.useCallback(() => {
        setEditModalOpen(true);
    }, []);

    const onEditModalClose = React.useCallback(() => {
        setEditModalOpen(false);
    }, []);

    const locale = getI18nLocale();

    const hasData = React.useMemo(
        () => passportPhotos && Boolean(Object.keys(passportPhotos).length),
        [passportPhotos],
    );

    const photos = React.useMemo(() => ({ ...EMPTY_PHOTOS, ...passportPhotos }), [passportPhotos]);
    const details = React.useMemo((): UserDocumentDetailsItem[] => {
        if (!passportDetails) {
            return [];
        }

        return [
            {
                label: getUserDocumentPassportField('first_name'),
                value: passportDetails.first_name,
            },

            {
                label: getUserDocumentPassportField('last_name'),
                value: passportDetails.last_name,
            },

            {
                label: getUserDocumentPassportField('gender'),
                value: getUserDocumentPassportGender(passportDetails.gender),
            },

            {
                label: getUserDocumentPassportField('birth_date'),
                value: getDateString(passportDetails.birth_date, locale, USER_DOCUMENTS_DATE_FORMAT),
            },

            {
                label: getUserDocumentPassportField('citizenship'),
                value: passportDetails.citizenship,
            },

            {
                label: getUserDocumentPassportField('doc_value'),
                value: passportDetails.doc_value,
            },

            {
                label: getUserDocumentPassportField('issue_date'),
                value: getDateString(passportDetails.issue_date, locale, USER_DOCUMENTS_DATE_FORMAT),
            },
        ];
    }, [locale, passportDetails]);

    return (
        <>
            <UserDocumentsItem
                className={className}
                title={i18n('Passport/ID card')}
                submittedAt={submittedAt}
                updatedAt={updatedAt}
                photos={photos}
                details={details}
                onEdit={hasData ? onEditModalOpen : undefined}
            />

            {isEditModalOpen && (
                <UserEditDocumentPassportModal
                    resource={resource}
                    onClose={onEditModalClose}
                />
            )}
        </>
    );
};
