import * as React from 'react';
import cn from 'classnames';

import { getI18nLocale } from 'utils/language/getI18nLocale';

import { USER_DOCUMENTS_DATE_FORMAT } from 'features/UserDocuments/consts/UserDocumentsDateFormat';
import {
    UserDocumentDetails,
    UserDocumentDetailsItem,
} from 'features/UserDocuments/ui/UserDocumentDetails/UserDocumentDetails';
import { UserDocumentPreview } from 'features/UserDocuments/ui/UserDocumentPreview/UserDocumentPreview';

import { UseUserDocumentPhotoRes } from 'entities/User/api/useUser/useUser';

import { ButtonColor } from 'shared/consts/ButtonColor';
import { ButtonSize } from 'shared/consts/ButtonSize';
import { getDateString } from 'shared/helpers/getDateString/getDateString';
import { Button } from 'shared/ui/Button/Button';
import { DotText } from 'shared/ui/DotText/DotText';
import { Widget } from 'shared/ui/Widget/Widget';

import { i18n } from 'features/UserDocuments/ui/UserDocumentsItem/UserDocumentsItem.i18n';

import EditIcon from 'shared/ui/Icons/images/edit-outline-16.inline.svg';

import styles from 'features/UserDocuments/ui/UserDocumentsItem/UserDocumentsItem.css';

export interface UserDocumentsItemProps {
    className?: string;
    title: string;
    submittedAt?: Date;
    updatedAt?: Date;
    photos: UseUserDocumentPhotoRes;
    details?: UserDocumentDetailsItem[];

    onEdit?(): void;
}

export const UserDocumentsItem: React.FC<UserDocumentsItemProps> = function UserDocumentsItem({
    className,
    title,
    submittedAt,
    updatedAt,
    photos,
    details,
    onEdit,
}) {
    const locale = getI18nLocale();

    const dates = React.useMemo(() => {
        if (submittedAt) {
            return [
                i18n('Added {date}', {
                    date: getDateString(submittedAt, locale, USER_DOCUMENTS_DATE_FORMAT),
                }),

                updatedAt
                    ? i18n('Last update {date}', {
                          date: getDateString(updatedAt, locale, USER_DOCUMENTS_DATE_FORMAT),
                      })
                    : null,
            ];
        }

        return [];
    }, [submittedAt, updatedAt]);

    return (
        <Widget className={cn(styles.item, className)}>
            <div className={styles.header}>
                <div className={styles.headerControls}>
                    <Button
                        color={ButtonColor.PRIMARY}
                        size={ButtonSize.M}
                        icon={EditIcon}
                        label={i18n('Edit data')}
                        disabled={!onEdit}
                        onClick={onEdit}
                    />
                </div>

                <div className={styles.headerText}>
                    <h3 className={styles.title}>{title}</h3>

                    {Boolean(dates.length) && (
                        <p className={styles.description}>
                            <DotText items={dates} />
                        </p>
                    )}
                </div>
            </div>

            <div className={styles.content}>
                <UserDocumentPreview photos={photos} />

                <UserDocumentDetails items={details} />
            </div>
        </Widget>
    );
};
