import * as React from 'react';
import cn from 'classnames/bind';

import { UserEditDocumentModalPicture } from 'features/UserDocuments/ui/UserEditDocumentModalPicture/UserEditDocumentModalPicture';

import { getUserDocumentPhotoData, UseUserDocumentPhotoRes } from 'entities/User';

import { ButtonColor } from 'shared/consts/ButtonColor';
import { ButtonSize } from 'shared/consts/ButtonSize';
import { Button } from 'shared/ui/Button/Button';
import { ModalContainer, ModalContainerProps } from 'shared/ui/ModalContainer/ModalContainer';

import { i18n } from 'features/UserDocuments/ui/UserEditDocumentModal/UserEditDocumentModal.i18n';

import styles from 'features/UserDocuments/ui/UserEditDocumentModal/UserEditDocumentModal.css';

export interface UserEditDocumentModalProps extends Pick<ModalContainerProps, 'onClose'> {
    photos: UseUserDocumentPhotoRes;

    onSubmit?(): void;
}

const cx = cn.bind(styles);

export const UserEditDocumentModal: React.FC<UserEditDocumentModalProps> = function UserEditDocumentModal({
    photos,
    onClose,
    onSubmit,
    children,
}) {
    const [currentPreview, setCurrentPreview] = React.useState<number>(0);

    const onClickHandler = React.useCallback(({ currentTarget }: React.MouseEvent<HTMLButtonElement>) => {
        setCurrentPreview(Number(currentTarget.value));
    }, []);

    const previews = getUserDocumentPhotoData(photos);
    const hasPreview = Boolean(previews.length);

    return (
        <ModalContainer
            className={cx(styles.content, { minWidth: hasPreview })}
            hasClose
            onClose={onClose}
        >
            {hasPreview && (
                <div className={styles.preview}>
                    {previews.map((preview, index) => (
                        <UserEditDocumentModalPicture
                            className={cx(styles.picture, { visible: currentPreview === index })}
                            {...preview}
                            key={preview.type}
                        />
                    ))}

                    <div className={styles.dots}>
                        {previews.map((preview, index) => (
                            <Button
                                className={cx(styles.dot, { current: currentPreview === index })}
                                value={index}
                                round
                                onClick={onClickHandler}
                                key={preview.type}
                            />
                        ))}
                    </div>
                </div>
            )}

            <div className={styles.form}>
                <h3 className={styles.title}>{i18n('Editing data')}</h3>

                <div className={styles.formContent}>{children}</div>

                <div className={styles.controls}>
                    <Button
                        color={ButtonColor.PRIMARY}
                        size={ButtonSize.M}
                        label={i18n('Save changes')}
                        disabled={!onSubmit}
                        wide
                        onClick={onSubmit}
                    />
                </div>
            </div>
        </ModalContainer>
    );
};
