import * as React from 'react';
import { useHistory, useParams, useRouteMatch } from 'react-router-dom';
import cn from 'classnames';

import { isShowUsersRides } from 'utils/isShowUsersRides';
import { isUserDocumentsPageFlag } from 'utils/isUserDocumentsPageFlag';

import { UseUserMemberResource } from 'entities/User/api/useUserMember/useUserMember';
import { hasUserRoleDriver } from 'entities/User/helpers/hasUserRoleDriver/hasUserRoleDriver';

import { Path } from 'shared/consts/Path';
import { TabSize } from 'shared/consts/TabSize';
import { TabItemOptions } from 'shared/ui/TabItem/TabItem';
import { Tabs } from 'shared/ui/Tabs/Tabs';

import { i18n } from 'features/UserHeaderTabs/ui/UserHeaderTabs/UserHeaderTabs.i18n';

import styles from 'features/UserHeaderTabs/ui/UserHeaderTabs/UserHeaderTabs.css';

enum UserHeaderTabsValue {
    PROFILE = 'profile',
    DOCUMENTS = 'documents',
    RIDES = 'rides',
}

export interface UserHeaderTabsProps {
    className?: string;
    resource: UseUserMemberResource;
}

// @todo: add storybook test
export const UserHeaderTabs: React.FC<UserHeaderTabsProps> = function UserHeaderTabs({ className, resource }) {
    const { id } = useParams<{ id: string }>();

    const history = useHistory();
    const routeMatch = useRouteMatch<{ tab: string }>(`${Path.USERS}/:id/:tab`);

    const user = resource.read();
    const hasDriverRole = hasUserRoleDriver(user.roles);
    const hasDocumentsTab = hasDriverRole && isUserDocumentsPageFlag();
    const hasRidesTab = hasDriverRole && isShowUsersRides();

    const [currentItem, setCurrentItem] = React.useState<string>();

    React.useEffect(() => {
        setCurrentItem(routeMatch?.params?.tab || UserHeaderTabsValue.PROFILE);
    }, [routeMatch]);

    const onItemClickHandler = React.useCallback(
        (value: UserHeaderTabsValue) => {
            setCurrentItem(value);

            switch (value) {
                case UserHeaderTabsValue.PROFILE:
                    // @todo: move to helper
                    history.push(`${Path.USERS}/${id}/profile`);
                    break;

                case UserHeaderTabsValue.DOCUMENTS:
                    // @todo: move to helper
                    history.push(`${Path.USERS}/${id}/documents`);
                    break;

                case UserHeaderTabsValue.RIDES:
                    // @todo: move to helper
                    history.push(`${Path.USERS}/${id}/rides`);
                    break;
            }
        },
        [id],
    );

    const items = React.useMemo(
        (): TabItemOptions[] => [
            { label: i18n('Profile'), value: UserHeaderTabsValue.PROFILE },
            ...(hasDocumentsTab ? [{ label: i18n('Documents'), value: UserHeaderTabsValue.DOCUMENTS }] : []),
            ...(hasRidesTab ? [{ label: i18n('Rides'), value: UserHeaderTabsValue.RIDES }] : []),
        ],

        [hasRidesTab],
    );

    if (items.length < 2) {
        return null;
    }

    return (
        <Tabs
            className={cn(styles.userHeaderTabs, className)}
            size={TabSize.M}
            items={items}
            currentItem={currentItem}
            onItemClick={onItemClickHandler}
        />
    );
};
