import * as React from 'react';
import { useCallback, useEffect, useState } from 'react';

import { TagDescription, useAffiliationTags } from 'entities/Tag/api/useAffiliationTags/useAffiliationTags';
import { deleteUserTag } from 'entities/User/api/deleteUserTag/deleteUserTag';
import { postUserTag, PostUserTagReq } from 'entities/User/api/postUserTag/postUserTag';
import { UserTag, useUserTags } from 'entities/User/api/useUserTags/useUserTags';

import { ButtonColor } from 'shared/consts/ButtonColor';
import { ButtonSize } from 'shared/consts/ButtonSize';
import { HttpStatusCode } from 'shared/consts/HttpStatusCode';
import { InputSize } from 'shared/consts/InputSize';
import { buildFetchRequestResult } from 'shared/helpers/fetchRequest/fetchRequest';
import { getFetchErrorMessage } from 'shared/helpers/getFetchErrorMessage/getFetchErrorMessage';
import { Button } from 'shared/ui/Button/Button';
import { ErrorMessage } from 'shared/ui/ErrorMessage/ErrorMessage';
import { Suggest, SuggestValueState } from 'shared/ui/Suggest/Suggest';

import { i18n } from 'features/UserOrganizationTags/ui/UserOrganizationTags/UserOrganizationTags.i18n';

import style from 'features/UserOrganizationTags/ui/UserOrganizationTags/UserOrganizationTags.css';

export interface UserOrganizationTagsProps {
    userId: string;
}

const MENU_MAX_HEIGHT = 300;

export const UserOrganizationTags: React.FC<UserOrganizationTagsProps> = function UserOrganizationTags({ userId }) {
    const { data: affiliationTags, error: tagsError, reload: reloadTags } = useAffiliationTags();
    const { data: userTags, error: userTagsError, reload: reloadUserTags } = useUserTags(userId ?? '');

    const [userAffiliationTag, setUserAffiliationTag] = useState<UserTag | null>(null);
    const [initialTag, setInitialTag] = useState<SuggestValueState<UserTag>>();
    const [selectedTag, setSelectedTag] = useState<UserTag>();
    const [error, setError] = useState<Error | null>(null);
    const [isSuccess, setSuccess] = useState<boolean>(false);

    const currentError = error || tagsError || userTagsError;

    const getSuggestOption = useCallback((tag?: UserTag | TagDescription): SuggestValueState<UserTag> => {
        return {
            text: tag?.display_name || tag?.name || tag?.tag || '',
            id: tag?.name || tag?.tag,
            data: tag,
        };
    }, []);

    useEffect(() => {
        if (userTags && affiliationTags) {
            const userAffiliationTag =
                userTags?.records?.find((el) => {
                    return affiliationTags.some((tag) => el.tag === tag.name);
                }) ?? ({} as UserTag);
            setUserAffiliationTag(userAffiliationTag);
            setInitialTag(getSuggestOption(userAffiliationTag));
        }
    }, [userTags, getSuggestOption, affiliationTags]);

    const onSuggestChange = useCallback(
        (tagName, tag) => {
            setSelectedTag(tag);
            isSuccess && setSuccess(false);
        },
        [isSuccess],
    );

    const deleteTag = React.useCallback(() => {
        try {
            userAffiliationTag?.tag_id && deleteUserTag(userAffiliationTag.tag_id);
        } catch (error) {
            if (error.status !== HttpStatusCode.OK) {
                setError(error);
            }
        }
    }, [userAffiliationTag?.tag_id]);

    const addTag = React.useCallback(() => {
        const fetchData: PostUserTagReq = {
            object_id: userId ?? '',
            tag: selectedTag?.name ?? '',
            roles: ['admin'],
            priority: 0,
        };

        postUserTag(fetchData)
            .then(() => {
                setSuccess(true);
                setInitialTag(getSuggestOption(selectedTag));
            })
            .catch((error) => {
                setError(error);
            });
    }, [userId, selectedTag, getSuggestOption]);

    const onSaveClick = React.useCallback(async () => {
        await addTag();

        if (initialTag) {
            await deleteTag();
        }

        reloadTags();
        reloadUserTags();
    }, [addTag, deleteTag, initialTag, reloadTags, reloadUserTags]);

    const dataProvider = useCallback(
        (text: string) => {
            if (text) {
                const filteredTags = affiliationTags?.filter((tag) => {
                    return (
                        tag.display_name?.toLocaleLowerCase()?.includes(text.toLowerCase()) ||
                        tag.name?.toLowerCase()?.includes(text)
                    );
                });

                return buildFetchRequestResult(Promise.resolve(filteredTags ?? []));
            }

            return buildFetchRequestResult(Promise.resolve(affiliationTags ?? []));
        },
        [affiliationTags],
    );

    const menuItemsProvider = useCallback((items) => {
        return (
            items?.map?.((tag) => {
                return { data: tag, text: tag.display_name || tag.name, id: tag.name };
            }) ?? []
        );
    }, []);

    return (
        <React.Suspense fallback={<div />}>
            <div className={style.affiliationTags}>
                {initialTag && (
                    <Suggest<UserTag>
                        inputSize={InputSize.M}
                        placeholder={i18n('Organization tags')}
                        dataProvider={dataProvider}
                        menuItemsProvider={menuItemsProvider}
                        initialValue={initialTag}
                        onSuggestChange={onSuggestChange}
                        maxHeight={MENU_MAX_HEIGHT}
                        data-testid="affiliationTags"
                    />
                )}

                {currentError ? <ErrorMessage error={getFetchErrorMessage(currentError)} /> : null}

                {isSuccess ? <div className={style.successLabel}>{i18n('The tag has been added')}</div> : null}

                <Button
                    color={ButtonColor.PRIMARY}
                    size={ButtonSize.M}
                    disabled={!selectedTag || initialTag?.id === selectedTag?.name}
                    onClick={onSaveClick}
                >
                    {i18n('Save')}
                </Button>
            </div>
        </React.Suspense>
    );
};
