import * as React from 'react';

import { ONE_MINUTE, ONE_MONTH, ONE_SECOND } from 'constants/constants';

import { showCarScoring } from 'utils/showCarScoring';

import { CarInformersOverview } from 'pages/Car/Overview/ui/CarInformersOverview/CarInformersOverview';
import { CarMaintenanceOverview } from 'pages/Car/Overview/ui/CarMaintenanceOverview/CarMaintenanceOverview';
import { CarOverview } from 'pages/Car/Overview/ui/CarOverview/CarOverview';

import { CarHardware } from 'features/CarHardware';
import { CarLatestRide } from 'features/CarLatestRide/ui/CarLatestRide/CarLatestRide';
import { CarLocation } from 'features/CarLocation';
import { CarScoring } from 'features/CarScoring';
import { CarSignals } from 'features/CarSignals';

import { UseCarResource } from 'entities/Car/api/useCar/useCar';
import { useCarBeacon } from 'entities/Car/api/useCarBeacon/useCarBeacon';
import { useCarRides } from 'entities/Car/api/useCarRides/useCarRides';
import { useCarSignals } from 'entities/Car/api/useCarSignals/useCarSignals';
import { useCarTelematics } from 'entities/Car/api/useCarTelematics/useCarTelematics';

import { DateFilter } from 'shared/consts/DateFilter';
import { getDateEndDay } from 'shared/helpers/getDateEndDay/getDateEndDay';
import { getDateStartDay } from 'shared/helpers/getDateStartDay/getDateStartDay';
import { useResourceReader } from 'shared/hooks/useResourceReader/useResourceReader';
import { useResourceReloader } from 'shared/hooks/useResourceReloader/useResourceReloader';
import { ContentContainer } from 'shared/ui/ContentContainer/ContentContainer';
import { ErrorBoundary } from 'shared/ui/ErrorBoundary/ErrorBoundary';
import { PageLoading } from 'shared/ui/PageLoading/PageLoading';
import { WidgetResourceBoundary } from 'shared/ui/WidgetResourceBoundary/WidgetResourceBoundary';

import styles from 'pages/Car/Overview/ui/CarLayoutOverview/CarLayoutOverview.css';

export interface CarLayoutOverviewProps {
    id: string;

    carResource: UseCarResource;
}

const FIVE = 5;
const FIVE_SECONDS = ONE_SECOND * FIVE;

export const CarLayoutOverview: React.FC<CarLayoutOverviewProps> = function CarLayoutOverview({ id, carResource }) {
    const car = useResourceReader(carResource, true);
    const carBeaconResource = useCarBeacon({ id });
    useResourceReloader(carBeaconResource, ONE_MINUTE);

    const hasDevice = car && Boolean(car.location || car.signalq);

    const carSignalsResource = useCarSignals(
        {
            id,
            [DateFilter.SINCE]: (getDateStartDay(new Date()).getTime() - ONE_MONTH) / ONE_SECOND,
            [DateFilter.UNTIL]: Math.trunc(getDateEndDay(new Date()).getTime() / ONE_SECOND),
        },
        !hasDevice,
    );
    const carRidesResource = useCarRides({ id }, !hasDevice);
    const carTelematicsResource = useCarTelematics({ id }, car && !car.location);
    useResourceReloader(carTelematicsResource, FIVE_SECONDS);

    const onHardwareChange = React.useCallback(() => {
        carResource.reload();
    }, [carResource]);

    if (!car) {
        return <PageLoading />;
    }

    if (hasDevice) {
        if (car.location) {
            // car + telematics
            // car + telematics + beacon
            return (
                <ContentContainer
                    className={styles.content}
                    bodyScroll
                >
                    <div className={styles.column}>
                        <CarLocation
                            resource={carResource}
                            beaconResource={carBeaconResource}
                        />

                        <WidgetResourceBoundary>
                            <CarHardware
                                resource={carResource}
                                beaconResource={carBeaconResource}
                                carTelematicsResource={carTelematicsResource}
                            />
                        </WidgetResourceBoundary>

                        {showCarScoring() && <CarScoring resource={carResource} />}
                    </div>
                    <div className={styles.column}>
                        <WidgetResourceBoundary>
                            <CarSignals resource={carSignalsResource} />
                        </WidgetResourceBoundary>

                        <WidgetResourceBoundary>
                            <CarLatestRide resource={carRidesResource} />
                        </WidgetResourceBoundary>

                        <CarInformersOverview resource={carResource} />

                        <CarMaintenanceOverview resource={carResource} />
                    </div>
                </ContentContainer>
            );
        }

        if (car.signalq) {
            // car + camera
            // car + camera + beacon
            return (
                <ContentContainer
                    className={styles.content}
                    bodyScroll
                >
                    <div className={styles.column}>
                        <WidgetResourceBoundary>
                            <CarLocation
                                resource={carResource}
                                beaconResource={carBeaconResource}
                            />
                        </WidgetResourceBoundary>

                        <WidgetResourceBoundary>
                            <CarHardware
                                resource={carResource}
                                beaconResource={carBeaconResource}
                                // carTelematicsResource={carTelematicsResource} // @TODO пока без варианта камера + телематика
                                onHardwareChange={onHardwareChange}
                            />
                        </WidgetResourceBoundary>

                        <CarMaintenanceOverview resource={carResource} />
                    </div>
                    <div className={styles.column}>
                        <WidgetResourceBoundary>
                            <CarSignals resource={carSignalsResource} />
                        </WidgetResourceBoundary>

                        <WidgetResourceBoundary>
                            <CarLatestRide resource={carRidesResource} />
                        </WidgetResourceBoundary>
                    </div>
                </ContentContainer>
            );
        }
    }

    return (
        <ContentContainer
            className={styles.content}
            bodyScroll
        >
            <ErrorBoundary
                fallback={
                    <CarOverview
                        className={styles.column}
                        carResource={carResource}
                        onHardwareChange={onHardwareChange}
                    />
                }
            >
                <CarOverview
                    className={styles.column}
                    carResource={carResource}
                    beaconResource={carBeaconResource}
                    onHardwareChange={onHardwareChange}
                />
            </ErrorBoundary>
        </ContentContainer>
    );
};
