import * as React from 'react';
import { useState } from 'react';

import { Header } from 'widgets/Header';
import { RentalRetiree } from 'widgets/RentalRetiree';

import { CarAddBookingModal, CarEditBookingModal } from 'features/CarBooking';
import { CarAddServiceModal, CarEditServiceModal } from 'features/CarService';
import { CarsHeaderTabs } from 'features/CarsHeaderTabs';
import { CarsScheduleGrid, CarsScheduleGridCreateCallback } from 'features/CarsScheduleGrid';
import { CarsSchedulePopup } from 'features/CarsSchedulePopup';

import { CarRentalRetireeStepSchema, CarsScheduleList } from 'entities/Car';
import { useCarList, UseCarListData } from 'entities/Car/api/useCarList/useCarList';
import { useCarsOffersTimetable } from 'entities/Car/api/useCarsOffersTimetable/useCarsOffersTimetable';
import { CarsScheduleFormType } from 'entities/Car/consts/CarsScheduleFormType';
import { CarOfferPreviewSchema } from 'entities/Car/types/CarOfferPreviewSchema';
import { CarSearchSchema } from 'entities/Car/types/CarSearchSchema';
import { CarsScheduleListItems } from 'entities/Car/ui/CarsScheduleList/CarsScheduleList';

import { formatTime } from 'shared/helpers/formatTime/formatTime';
import { getDateStartDay } from 'shared/helpers/getDateStartDay/getDateStartDay';
import { FormDateTimeRangeSchema } from 'shared/types/FormDateTimeRangeSchema';
import { ContentContainer } from 'shared/ui/ContentContainer/ContentContainer';
import { SectionLayout } from 'shared/ui/SectionLayout/SectionLayout';

import { i18n } from 'pages/Cars/Schedule/ui/CarsSchedulePage/CarsSchedulePage.i18n';

import styles from 'pages/Cars/Schedule/ui/CarsSchedulePage/CarsSchedulePage.css';

const LIST_PAYLOAD: UseCarListData = {
    traits: 'ReportVIN',
    page_size: 50,
};

export interface CarsScheduleCreateFormSchema extends FormDateTimeRangeSchema {
    car: CarSearchSchema;
}

interface CarsScheduleRentalData {
    offerId: string;
    tagId: string;
    userId: string;
    initialStep: CarRentalRetireeStepSchema;
}

// @todo: add storybook test
export const CarsSchedulePage: React.FC = function CarsSchedulePage() {
    const itemRef = React.useRef() as React.MutableRefObject<Nullable<HTMLDivElement>>;

    const [ver, setVer] = useState<number>(0);
    const [hoveredCarId, setHoveredCarId] = React.useState<Optional<string>>();
    const [currentItem, setCurrentItem] = React.useState<Optional<CarOfferPreviewSchema>>();
    const [createData, setCreateData] = React.useState<Optional<Partial<CarsScheduleCreateFormSchema>>>();
    const [editData, setEditData] = React.useState<Optional<CarOfferPreviewSchema>>();
    const [formType, setFormType] = React.useState<CarsScheduleFormType>();
    const [rentalData, setRentalData] = React.useState<Optional<CarsScheduleRentalData>>();

    const timeRange = React.useMemo(() => {
        const now = new Date();
        const since = new Date(now.getFullYear(), now.getMonth() - 1, 1, 0, 0, 0, 0);
        const until = new Date(now.getFullYear() + 1, now.getMonth(), 1, 0, 0, 0, 0);

        return { since, until };
    }, []);

    const { data, isFullyLoaded, isLoading, fetchNextPage } = useCarList(LIST_PAYLOAD);
    const { data: dataOffers, reload } = useCarsOffersTimetable(timeRange);

    const carsList = React.useMemo(
        (): CarsScheduleListItems => ({
            cars: data?.cars.map(({ id, model_id, number, vin }) => ({
                id,
                car_details: {
                    id,
                    model_id,
                    model_name: data?.models?.[model_id]?.name || model_id,
                    number,
                    vin,
                },

                offers: dataOffers?.[id],
            })),

            isFullyLoaded,
        }),

        [dataOffers, data, isFullyLoaded],
    );

    const onMouseEnterItemHandler = React.useCallback(
        ({ currentTarget }: React.MouseEvent<HTMLLIElement>) => {
            const id = currentTarget.getAttribute('data-id') || '';

            if (hoveredCarId !== id) {
                setHoveredCarId(id);
            }
        },
        [hoveredCarId],
    );

    const onMouseLeaveHandler = React.useCallback(() => {
        setHoveredCarId(undefined);
    }, []);

    const onCreateHandler = React.useCallback<CarsScheduleGridCreateCallback>(
        (type, payload) => {
            if (!payload) {
                setFormType(type);
                setCreateData({});

                return;
            }

            const { id, since, until } = payload;

            const car = carsList.cars?.find((car) => car.id === id);

            if (car) {
                setFormType(type);
                setCreateData({
                    car: car.car_details,
                    since: getDateStartDay(since),
                    since_time: formatTime(since),
                    until: getDateStartDay(until),
                    until_time: formatTime(until),
                });
            }
        },
        [carsList],
    );

    const onItemClickHandler = React.useCallback(
        (data: CarOfferPreviewSchema, ref: React.MutableRefObject<Nullable<HTMLDivElement>>) => {
            if (itemRef.current !== ref.current) {
                itemRef.current = ref.current;

                setVer((ver) => ver + 1);
            }

            setCurrentItem(data);
        },
        [],
    );

    const onFormCloseHandler = React.useCallback(() => {
        setCreateData(undefined);
        setEditData(undefined);
        setFormType(undefined);
    }, []);

    const onEditHandler = React.useCallback(
        (type: CarsScheduleFormType) => {
            if (currentItem) {
                setEditData(currentItem);
                setFormType(type);
            }
        },
        [currentItem],
    );

    const onPopupCloseHandler = React.useCallback(() => {
        setCurrentItem(undefined);

        itemRef.current = null;
    }, []);

    const onRentalSubmitHandler = React.useCallback(
        (formStep: CarRentalRetireeStepSchema) => {
            if (currentItem && currentItem.user) {
                setRentalData({
                    offerId: currentItem.offer_id,
                    tagId: currentItem.tag_id,
                    userId: currentItem.user.id,
                    initialStep: formStep,
                });
            }
        },
        [currentItem],
    );

    const onStartHandler = React.useCallback(
        (step: CarRentalRetireeStepSchema) => {
            onRentalSubmitHandler(step);
        },
        [onRentalSubmitHandler],
    );

    const onFinishHandler = React.useCallback(() => {
        onRentalSubmitHandler('finish');
    }, [onRentalSubmitHandler]);

    const onRentalCloseHandler = React.useCallback(() => {
        setRentalData(undefined);
    }, []);

    return (
        <SectionLayout
            header={
                <Header
                    title={i18n('Cars')}
                    tabs={<CarsHeaderTabs />}
                    withoutBorder
                />
            }
            bodyScroll
        >
            <ContentContainer
                className={styles.content}
                bodyScroll
            >
                <CarsScheduleList
                    className={styles.side}
                    items={carsList}
                    hoveredItem={hoveredCarId}
                    isLoading={isLoading}
                    onMouseEnterItem={onMouseEnterItemHandler}
                    onMouseLeaveItem={onMouseLeaveHandler}
                    fetchNextPage={fetchNextPage}
                />

                <CarsScheduleGrid
                    {...timeRange}
                    className={styles.grid}
                    items={carsList}
                    hoveredItem={hoveredCarId}
                    onCreate={onCreateHandler}
                    onItemClick={onItemClickHandler}
                    onMouseEnterItem={onMouseEnterItemHandler}
                    onMouseLeaveItem={onMouseLeaveHandler}
                />

                {createData && (
                    <>
                        {formType === CarsScheduleFormType.BOOKING && (
                            <CarAddBookingModal
                                initial={createData}
                                onChange={reload}
                                onClose={onFormCloseHandler}
                            />
                        )}

                        {formType === CarsScheduleFormType.SERVICE && (
                            <CarAddServiceModal
                                initial={createData}
                                onChange={reload}
                                onClose={onFormCloseHandler}
                            />
                        )}
                    </>
                )}

                {currentItem && (
                    <CarsSchedulePopup
                        offerId={currentItem.offer_id}
                        tagId={currentItem.tag_id}
                        user={currentItem.user}
                        status={currentItem.status}
                        onEdit={onEditHandler}
                        onStart={onStartHandler}
                        onFinish={onFinishHandler}
                        onClose={onPopupCloseHandler}
                        referenceRef={itemRef}
                        key={ver}
                    />
                )}

                {editData && (
                    <>
                        {formType === CarsScheduleFormType.BOOKING && (
                            <CarEditBookingModal
                                offerId={editData.offer_id}
                                tagId={editData.tag_id}
                                user={editData.user}
                                onChange={reload}
                                onClose={onFormCloseHandler}
                            />
                        )}

                        {formType === CarsScheduleFormType.SERVICE && (
                            <CarEditServiceModal
                                tagId={editData.tag_id}
                                car={editData.car}
                                onChange={reload}
                                onClose={onFormCloseHandler}
                            />
                        )}
                    </>
                )}

                {rentalData && (
                    <RentalRetiree
                        {...rentalData}
                        onChange={reload}
                        onClose={onRentalCloseHandler}
                    />
                )}
            </ContentContainer>
        </SectionLayout>
    );
};
