import * as React from 'react';
import { useHistory, useLocation } from 'react-router-dom';

import { ONE_SECOND, ONE_WEEK } from 'constants/constants';

import { isCarDashboardScoringFlag } from 'utils/isCarDashboardScoring';

import { DashboardFilter } from 'pages/Dashboard/Dashboard/conts/DashboardFilter';
import { EfficiencyBoard } from 'pages/Dashboard/Dashboard/ui/EfficiencyBoard/EfficiencyBoard';
import { StateBoard } from 'pages/Dashboard/Dashboard/ui/StateBoard/StateBoard';

import { Header } from 'widgets/Header';

import { DashboardFleetScoring } from 'features/DashboardFleetScoring/ui/DashboardFleetScoring/DashboardFleetScoring';
import { FilterDate } from 'features/FilterDate/ui/FilterDate/FilterDate';

import { useRidesSummary } from 'entities/Ride/api/useRidesSummary/useRidesSummary';

import { DateFilter } from 'shared/consts/DateFilter';
import { getDateAggregation } from 'shared/helpers/getDateAggregation/getDateAggregation';
import { getDateEndDay } from 'shared/helpers/getDateEndDay/getDateEndDay';
import { getDateStartDay } from 'shared/helpers/getDateStartDay/getDateStartDay';
import { getFetchErrorMessage } from 'shared/helpers/getFetchErrorMessage/getFetchErrorMessage';
import { updateFiltersParams } from 'shared/helpers/updateFiltersParams/updateFiltersParams';
import { useRetry } from 'shared/hooks/useRetry/useRetry';
import { useSearchFilters } from 'shared/hooks/useSearchFilters/useSearchFilters';
import { ContentContainer } from 'shared/ui/ContentContainer/ContentContainer';
import { ErrorBoundary, ErrorBoundaryFallbackProps } from 'shared/ui/ErrorBoundary/ErrorBoundary';
import { ErrorMessage } from 'shared/ui/ErrorMessage/ErrorMessage';
import { PageLoading } from 'shared/ui/PageLoading/PageLoading';
import { SectionLayout } from 'shared/ui/SectionLayout/SectionLayout';

import { RidesFiltersOptions } from 'components/Rides';

import { i18n } from 'pages/Dashboard/Dashboard/ui/Dashboard/Dashboard.i18n';

import styles from 'pages/Dashboard/Dashboard/ui/Dashboard/Dashboard.css';

export interface DashboardProps {
    className?: string;
}

export const Dashboard: React.FC<DashboardProps> = function Dashboard({ className }) {
    const location = useLocation();
    const history = useHistory();

    // @todo: moved into component because storybook set mock date only for one story; set mock date for all app
    const initialDashboardFilters = {
        [DateFilter.SINCE]: (getDateStartDay(new Date()).getTime() - ONE_WEEK * 2) / ONE_SECOND,
        [DateFilter.UNTIL]: Math.trunc(getDateEndDay(new Date()).getTime() / ONE_SECOND),
    };
    const existingDashboardFilters = [DateFilter.SINCE, DateFilter.UNTIL];
    const dateDashboardFilters = [DateFilter.SINCE, DateFilter.UNTIL];

    const filters = useSearchFilters<RidesFiltersOptions>(initialDashboardFilters, existingDashboardFilters, {
        dateFilters: dateDashboardFilters,
    });

    const aggregationConfig = getDateAggregation({
        since: filters[DateFilter.SINCE],
        until: filters[DateFilter.UNTIL],
    });

    const {
        data: summary,
        isLoading: isSummaryLoading,
        error,
        reload,
    } = useRidesSummary({
        since: filters[DateFilter.SINCE],
        until: filters[DateFilter.UNTIL],
        aggregation: aggregationConfig.aggregation,
        tariff: undefined,
    });
    const reachRetryLimit = useRetry(error, reload);
    const { totalCars, statuses } = summary || {};

    const onChangeFilterHandler = React.useCallback(
        (type: string, value: Record<DateFilter, Nullable<number>>) => {
            const searchParams = new URLSearchParams(location?.search);

            history.push(`${location.pathname}?${updateFiltersParams(searchParams, type, value)}`);
        },
        [location],
    );

    if (reachRetryLimit && error) {
        return (
            <SectionLayout
                header={
                    <Header
                        title={i18n('Dashboard')}
                        withoutBorder
                    />
                }
                bodyScroll
            >
                <ContentContainer
                    className={styles.content}
                    bodyScroll
                >
                    <h3 className={styles.title}>
                        {i18n("Try to reload the page, if it doesn't help - try to come back later")}
                    </h3>
                    <ErrorMessage error={getFetchErrorMessage(error)} />
                </ContentContainer>
            </SectionLayout>
        );
    }

    const Fallback: React.FC<ErrorBoundaryFallbackProps> = React.useCallback(function Fallback({ error }) {
        return (
            <div className={styles.content}>
                <h3 className={styles.title}>{i18n('Something went wrong')}</h3>
                <ErrorMessage error={getFetchErrorMessage(error)} />
            </div>
        );
    }, []);

    return (
        <ErrorBoundary fallback={Fallback}>
            <SectionLayout
                header={
                    <Header
                        title={i18n('Dashboard')}
                        withoutBorder
                    />
                }
                bodyScroll
            >
                <React.Suspense fallback={<PageLoading />}>
                    <ContentContainer bodyScroll>
                        <div>
                            <div className={styles.filters}>
                                <h2 className={styles.rowTitle}>{i18n('Realtime states')}</h2>

                                <FilterDate
                                    className={styles.filterDate}
                                    type={DashboardFilter.DATE}
                                    filters={filters}
                                    onChangeFilter={onChangeFilterHandler}
                                    placement="bottom-end"
                                    inline
                                />
                            </div>

                            <StateBoard
                                statuses={statuses}
                                totalCars={totalCars}
                                isLoading={isSummaryLoading}
                            />
                        </div>

                        <div className={styles.row}>
                            <h2 className={styles.rowTitle}>{i18n('Utilization and efficiency')}</h2>

                            <EfficiencyBoard
                                isLoading={isSummaryLoading}
                                summary={summary}
                                aggregationConfig={aggregationConfig}
                            />
                        </div>

                        {isCarDashboardScoringFlag() && (
                            <div className={styles.row}>
                                <h2 className={styles.rowTitle}>{i18n('Behavior and security')}</h2>

                                <DashboardFleetScoring aggregationConfig={aggregationConfig} />
                            </div>
                        )}
                    </ContentContainer>
                </React.Suspense>
            </SectionLayout>
        </ErrorBoundary>
    );
};
