import * as React from 'react';
import cn from 'classnames';

import { formatCarNumber } from 'entities/Car';
import { CarTop, UseRidesSummaryRes } from 'entities/Ride/api/useRidesSummary/useRidesSummary';

import { IslandTheme } from 'shared/consts/IslandTheme';
import { Path } from 'shared/consts/Path';
import { formatHours } from 'shared/helpers/formatHours/formatHours';
import { formatMileage } from 'shared/helpers/formatMileage/formatMileage';
import { formatNumber } from 'shared/helpers/formatNumber/formatNumber';
import { formatUtilization } from 'shared/helpers/formatUtilization/formatUtilization';
import { generateRouterPath } from 'shared/helpers/generateRouterPath/generateRouterPath';
import { DateAggregationConfig } from 'shared/helpers/getDateAggregation/getDateAggregation';
import { getDateAggregationLabel } from 'shared/helpers/getDateAggregationLabel/getDateAggregationLabel';
import { ChartKey } from 'shared/types/ChartPoint';
import { ChartViewHistogram } from 'shared/ui/ChartViewHistogram/ChartViewHistogram';
import { Island } from 'shared/ui/Island/Island';
import { Link } from 'shared/ui/Link/Link';
import { Spin } from 'shared/ui/Spin/Spin';
import { TopTable, TopTableItem } from 'shared/ui/TopTable/TopTable';

import { i18n } from 'pages/Dashboard/Dashboard/ui/EfficiencyBoard/EfficiencyBoard.i18n';

import styles from 'pages/Dashboard/Dashboard/ui/EfficiencyBoard/EfficiencyBoard.css';

export interface EfficiencyBoardProps {
    className?: string;

    summary?: UseRidesSummaryRes;
    isLoading: boolean;
    aggregationConfig: DateAggregationConfig;
}

const CHART_KEYS: ChartKey[] = ['line1'];

function getTopTableItems(top: Optional<CarTop[]>, isPercent?: boolean): TopTableItem[] {
    if (!top) {
        return [];
    }

    return top.map((car, index): TopTableItem => {
        const value = isPercent ? car.value * 100 : car.value;
        const formattedValue = formatNumber(value, {
            minimumFractionDigits: isPercent ? 1 : 0,
            maximumFractionDigits: isPercent ? 1 : 0,
        });

        return {
            id: car.object_id,
            text: (
                <Link
                    href={generateRouterPath(Path.CAR, { id: car.object_id })}
                    className={styles.car}
                >
                    <div className={styles.carName}>{car.name}</div>
                    <div className={styles.carNumber}>{formatCarNumber(car.number)}</div>
                </Link>
            ),

            value: isPercent ? `${formattedValue} %` : formattedValue,
        };
    });
}

export const EfficiencyBoard: React.FC<EfficiencyBoardProps> = function EfficiencyBoard({
    className,
    summary,
    isLoading,
    aggregationConfig,
}) {
    const AveragePerTime = getDateAggregationLabel(aggregationConfig.aggregation);

    const {
        topMileageCars,
        worseMileageCars,
        topHoursCars,
        worseHoursCars,
        topUtilizationCars,
        worseUtilizationCars,

        mileageChartPoints = [],
        hoursChartPoints = [],
        utilizationChartPoints = [],

        totalMileage,
        totalDuration,
        averageDurationByTime,
        averageMileageByTime,
        averageFleetUtilization,
    } = summary || {};

    return (
        <div className={cn(styles.container, [className])}>
            <div className={styles.ratingContainer}>
                <TopTable
                    title={i18n('Top cars by mileage')}
                    header={[i18n('Car'), i18n('km')]}
                    items={getTopTableItems(topMileageCars)}
                    isLoading={isLoading}
                />

                <TopTable
                    title={i18n('Worst cars by mileage')}
                    header={[i18n('Car'), i18n('km')]}
                    items={getTopTableItems(worseMileageCars)}
                    isLoading={isLoading}
                />

                <TopTable
                    title={i18n('Top cars by eng. hours')}
                    header={[i18n('Car'), i18n('h')]}
                    items={getTopTableItems(topHoursCars)}
                    isLoading={isLoading}
                />

                <TopTable
                    title={i18n('Worst cars by eng. hours')}
                    header={[i18n('Car'), i18n('h')]}
                    items={getTopTableItems(worseHoursCars)}
                    isLoading={isLoading}
                />
            </div>

            <div className={styles.chartsContainer}>
                <Island
                    className={styles.totalMileage}
                    title={i18n('Total mileage, km')}
                >
                    <div className={styles.tabletRow}>
                        <Island
                            className={styles.tablet}
                            title={
                                <h3 className={styles.islandSubTitle}>
                                    {isLoading ? <Spin /> : formatMileage(totalMileage)}
                                </h3>
                            }
                            theme={IslandTheme.GRAY}
                        >
                            {i18n('Total for the period')}
                        </Island>

                        <Island
                            className={styles.tablet}
                            title={
                                <h3 className={styles.islandSubTitle}>
                                    {isLoading ? <Spin /> : formatMileage(averageMileageByTime)}
                                </h3>
                            }
                            theme={IslandTheme.GRAY}
                        >
                            {AveragePerTime}
                        </Island>
                    </div>

                    {!isLoading ? (
                        <ChartViewHistogram
                            className={styles.chartMileage}
                            points={mileageChartPoints}
                            aggregationConfig={aggregationConfig}
                            keys={CHART_KEYS}
                            keysFormat={[
                                {
                                    fraction: 0,
                                    postfix: i18n('km'),
                                },
                            ]}
                        />
                    ) : (
                        <div className={styles.chartShimmer}>
                            <Spin />
                        </div>
                    )}
                </Island>

                <Island
                    className={styles.totalHours}
                    title={i18n('Total engine hours')}
                >
                    <div className={styles.tabletRow}>
                        <Island
                            className={styles.tablet}
                            title={
                                <h3 className={styles.islandSubTitle}>
                                    {isLoading ? <Spin /> : formatHours(totalDuration)}
                                </h3>
                            }
                            theme={IslandTheme.GRAY}
                        >
                            {i18n('Total for the period')}
                        </Island>

                        <Island
                            className={styles.tablet}
                            title={
                                <h3 className={styles.islandSubTitle}>
                                    {isLoading ? <Spin /> : formatHours(averageDurationByTime)}
                                </h3>
                            }
                            theme={IslandTheme.GRAY}
                        >
                            {AveragePerTime}
                        </Island>
                    </div>

                    {!isLoading && hoursChartPoints ? (
                        <ChartViewHistogram
                            className={styles.chartHours}
                            points={hoursChartPoints}
                            aggregationConfig={aggregationConfig}
                            keys={CHART_KEYS}
                            keysFormat={[
                                {
                                    fraction: 0,
                                    postfix: i18n('h'),
                                },
                            ]}
                        />
                    ) : (
                        <div className={styles.chartShimmer}>
                            <Spin />
                        </div>
                    )}
                </Island>
            </div>

            <div className={styles.utilizationContainer}>
                <TopTable
                    title={i18n('Top cars by utilisation')}
                    header={[i18n('Car'), '%']}
                    items={getTopTableItems(topUtilizationCars, true)}
                    isLoading={isLoading}
                />

                <TopTable
                    title={i18n('Worst cars by utilisation')}
                    header={[i18n('Car'), '%']}
                    items={getTopTableItems(worseUtilizationCars, true)}
                    isLoading={isLoading}
                />

                <Island
                    className={styles.utilizationChartIsland}
                    title={i18n('Engine hours utilisation, %')}
                >
                    <div className={styles.tabletRow}>
                        <Island
                            className={styles.tablet}
                            title={
                                <h3 className={styles.islandSubTitle}>
                                    {isLoading ? <Spin /> : formatUtilization(averageFleetUtilization)}
                                </h3>
                            }
                            theme={IslandTheme.GRAY}
                        >
                            {AveragePerTime}
                        </Island>
                    </div>

                    {!isLoading ? (
                        <ChartViewHistogram
                            className={styles.chartUtilization}
                            points={utilizationChartPoints}
                            aggregationConfig={aggregationConfig}
                            keys={CHART_KEYS}
                            keysFormat={[
                                {
                                    fraction: 1,
                                    postfix: '%',
                                },
                            ]}
                        />
                    ) : (
                        <div className={styles.chartShimmer}>
                            <Spin />
                        </div>
                    )}
                </Island>
            </div>
        </div>
    );
};
