import * as React from 'react';
import { useHistory } from 'react-router-dom';
import cn from 'classnames/bind';

import { CarStatusType } from 'entities/Car/consts/CarStatusType';
import { CarsView } from 'entities/Car/consts/CarsView';
import { CARS_VIEW_CGI } from 'entities/Car/consts/constants';
import { getCarStatus } from 'entities/Car/helpers/getCarStatus/getCarStatus';

import { IslandTheme } from 'shared/consts/IslandTheme';
import { Path } from 'shared/consts/Path';
import { TextWithDotSize } from 'shared/consts/TextWithDotSize';
import { Island } from 'shared/ui/Island/Island';
import { Spin } from 'shared/ui/Spin/Spin';
import { TextWithDot } from 'shared/ui/TextWithDot/TextWithDot';

import { i18n } from 'pages/Dashboard/Dashboard/ui/StateBoard/StateBoard.i18n';

import styles from 'pages/Dashboard/Dashboard/ui/StateBoard/StateBoard.css';

export interface StateBoardProps {
    className?: string;

    statuses?: Record<CarStatusType, number>;
    totalCars?: number;
    isLoading: boolean;
}

const cx = cn.bind(styles);

const STATUS_ORDER: CarStatusType[] = [
    CarStatusType.RIDING,
    CarStatusType.ENGINE_ON,
    CarStatusType.ENGINE_OFF,
    CarStatusType.NO_SIGNAL,
    CarStatusType.INSTALLATION,
    CarStatusType.UNINSTALL_TELEMATICS,
    CarStatusType.DELAYED_BLOCK,
    CarStatusType.DELAYED_UNBLOCK,
    CarStatusType.BLOCKED,
    CarStatusType.UNKNOWN,
];

export const StateBoard: React.FC<StateBoardProps> = function StateBoard({
    className,
    statuses,
    totalCars,
    isLoading,
}) {
    const history = useHistory();

    const [hoveredCarId, setHoveredCarId] = React.useState<Optional<string>>();
    const onMouseEnterItemHandler = React.useCallback(
        ({ currentTarget }: React.MouseEvent<HTMLDivElement>) => {
            const id = currentTarget.getAttribute('data-id') || '';

            if (hoveredCarId !== id) {
                setHoveredCarId(id);
            }
        },
        [hoveredCarId],
    );

    const onMouseLeaveHandler = React.useCallback(() => {
        setHoveredCarId(undefined);
    }, []);

    const onClick = React.useCallback(() => {
        if (hoveredCarId) {
            const searchParams = new URLSearchParams();

            searchParams.set(CARS_VIEW_CGI, CarsView.TABLE);
            searchParams.set('status', hoveredCarId);
            history.push(`${Path.CARS}?${searchParams}`);
        }
    }, [history, hoveredCarId]);

    return (
        <>
            <Island
                title={
                    <div className={styles.islandTitleContainer}>
                        <h3 className={styles.islandTitle}>
                            {isLoading ? (
                                <Spin />
                            ) : (
                                i18n('{count} car', {
                                    count: totalCars || 0,
                                })
                            )}
                        </h3>
                        <div className={styles.now}>{i18n('Now')}</div>
                    </div>
                }
            >
                <div className={styles.timeline}>
                    {totalCars && statuses ? (
                        STATUS_ORDER.map((status) => {
                            const data = statuses[status];

                            return data > 0 ? (
                                <div
                                    key={status}
                                    data-id={status}
                                    className={cx(styles.carStatusTimeline, { hovered: status === hoveredCarId }, [
                                        status,
                                    ])}
                                    onMouseEnter={onMouseEnterItemHandler}
                                    onMouseLeave={onMouseLeaveHandler}
                                    onClick={onClick}
                                    style={{
                                        flexBasis: `${(data / totalCars) * 100}%`,
                                    }}
                                />
                            ) : null;
                        })
                    ) : (
                        <div className={cx(styles.carStatusTimeline)} />
                    )}
                </div>

                <div className={styles.rowIslands}>
                    {STATUS_ORDER.map((status) => {
                        const data = statuses && statuses[status];

                        if (!data && status === CarStatusType.UNKNOWN) {
                            return null;
                        }

                        return (
                            <Island
                                key={status}
                                data-id={status}
                                className={cx(styles.islandNumber, { hovered: status === hoveredCarId })}
                                onMouseEnterItem={onMouseEnterItemHandler}
                                onMouseLeaveItem={onMouseLeaveHandler}
                                onClick={onClick}
                                title={
                                    !isLoading ? (
                                        <TextWithDot
                                            className={cx(styles.carStatus, [status])}
                                            size={TextWithDotSize.L}
                                            reverse
                                        >
                                            {data || 0}
                                        </TextWithDot>
                                    ) : (
                                        <div className={cx(styles.carStatusSpin)}>
                                            <Spin />
                                        </div>
                                    )
                                }
                                theme={IslandTheme.GRAY}
                            >
                                {getCarStatus(status)}
                            </Island>
                        );
                    })}
                </div>
            </Island>
        </>
    );
};
