import * as React from 'react';
import cn from 'classnames';

import { formatCarNumber } from 'entities/Car';
import { CarTop, UseRidesSummaryRes } from 'entities/Ride/api/useRidesSummary/useRidesSummary';

import { IslandTheme } from 'shared/consts/IslandTheme';
import { Path } from 'shared/consts/Path';
import { dataOrEmpty } from 'shared/helpers/dataOrEmpty/dataOrEmpty';
import { formatBookings } from 'shared/helpers/formatBookings/formatBookings';
import { formatDay } from 'shared/helpers/formatDay/formatDay';
import { formatNumber } from 'shared/helpers/formatNumber/formatNumber';
import { formatUtilization } from 'shared/helpers/formatUtilization/formatUtilization';
import { generateRouterPath } from 'shared/helpers/generateRouterPath/generateRouterPath';
import { DateAggregationConfig } from 'shared/helpers/getDateAggregation/getDateAggregation';
import { getDateAggregationLabel } from 'shared/helpers/getDateAggregationLabel/getDateAggregationLabel';
import { ChartKey } from 'shared/types/ChartPoint';
import { ChartViewHistogram } from 'shared/ui/ChartViewHistogram/ChartViewHistogram';
import { Island } from 'shared/ui/Island/Island';
import { Link } from 'shared/ui/Link/Link';
import { Spin } from 'shared/ui/Spin/Spin';
import { TopTable, TopTableItem } from 'shared/ui/TopTable/TopTable';

import { i18n } from 'pages/Dashboard/DashboardRental/ui/BookingEfficiencyBoard/BookingEfficiencyBoard.i18n';

import styles from 'pages/Dashboard/DashboardRental/ui/BookingEfficiencyBoard/BookingEfficiencyBoard.css';

export interface BookingEfficiencyBoardProps {
    className?: string;

    summary?: UseRidesSummaryRes;
    isLoading: boolean;
    aggregationConfig: DateAggregationConfig;
}

const CHART_KEYS: ChartKey[] = ['line1'];

function getTopTableItems(top: Optional<CarTop[]>, isPercent?: boolean): TopTableItem[] {
    if (!top) {
        return [];
    }

    return top.map((car, index): TopTableItem => {
        const value = isPercent ? car.value * 100 : car.value;
        const formattedValue = formatNumber(value, {
            minimumFractionDigits: isPercent ? 1 : 0,
            maximumFractionDigits: isPercent ? 1 : 0,
        });

        if (!car.object_id) {
            return {
                id: index,
                text: null,
                value: value,
            };
        }

        return {
            id: car.object_id,
            text: (
                <Link
                    href={generateRouterPath(Path.CAR, { id: car.object_id })}
                    className={styles.car}
                >
                    <div className={styles.carName}>{car.name}</div>
                    <div className={styles.carNumber}>{formatCarNumber(car.number)}</div>
                </Link>
            ),
            value: isPercent ? `${formattedValue} %` : formattedValue,
        };
    });
}

export const BookingEfficiencyBoard: React.FC<BookingEfficiencyBoardProps> = function BookingEfficiencyBoard({
    className,
    summary,
    isLoading,
    aggregationConfig,
}) {
    const AveragePerTime = getDateAggregationLabel(aggregationConfig.aggregation);

    const {
        rentals,
        bookingsAvgPerTime,
        bookingsTotalForThePeriod,
        rentalsAvgPerTime,

        averageRentalUtilization,

        topRentalUtilizationCars,
        worseRentalUtilizationCars,

        bookingsChartPoints = [],
        rentalUtilizationChartPoints = [],
    } = summary || {};

    return (
        <div className={cn(styles.container, [className])}>
            <div className={styles.column}>
                <Island title={i18n('Number of bookings')}>
                    <div className={styles.tabletRow}>
                        <Island
                            className={styles.tablet}
                            title={
                                <h3 className={styles.islandSubTitle}>
                                    {isLoading ? <Spin /> : formatBookings(bookingsTotalForThePeriod)}
                                </h3>
                            }
                            theme={IslandTheme.GRAY}
                        >
                            {i18n('Total for the period')}
                        </Island>

                        <Island
                            className={styles.tablet}
                            title={
                                <h3 className={styles.islandSubTitle}>
                                    {isLoading ? <Spin /> : formatBookings(bookingsAvgPerTime, 1)}
                                </h3>
                            }
                            theme={IslandTheme.GRAY}
                        >
                            {AveragePerTime}
                        </Island>
                    </div>

                    {!isLoading ? (
                        <ChartViewHistogram
                            className={styles.chartMileage}
                            points={bookingsChartPoints}
                            aggregationConfig={aggregationConfig}
                            keys={CHART_KEYS}
                            keysFormat={[
                                {
                                    fraction: 0,
                                },
                            ]}
                        />
                    ) : (
                        <div className={styles.chartShimmer}>
                            <Spin />
                        </div>
                    )}
                </Island>

                <div className={styles.rentalInto}>
                    <Island
                        className={styles.tablet}
                        title={
                            <h3 className={styles.islandSubTitle}>
                                {isLoading ? (
                                    <Spin />
                                ) : (
                                    dataOrEmpty(
                                        rentals,
                                        formatDay({
                                            value: formatNumber(rentals!, {
                                                minimumFractionDigits: 0,
                                                maximumFractionDigits: 0,
                                            }),
                                            count: rentals!,
                                        }),
                                    )
                                )}
                            </h3>
                        }
                    >
                        {i18n('Total rental days')}
                    </Island>
                    <Island
                        className={styles.tablet}
                        title={
                            <h3 className={styles.islandSubTitle}>
                                {isLoading ? (
                                    <Spin />
                                ) : (
                                    dataOrEmpty(
                                        rentalsAvgPerTime,
                                        formatDay({
                                            value: formatNumber(rentalsAvgPerTime!, {
                                                minimumFractionDigits: 1,
                                                maximumFractionDigits: 1,
                                            }),
                                            count: rentalsAvgPerTime!,
                                        }),
                                    )
                                )}
                            </h3>
                        }
                    >
                        {i18n('Avg. booking duration')}
                    </Island>
                </div>
            </div>

            <div className={styles.column}>
                <Island title={i18n('Utilisation')}>
                    <div className={styles.tabletRow}>
                        <Island
                            className={styles.tablet}
                            title={
                                <h3 className={styles.islandSubTitle}>
                                    {isLoading ? <Spin /> : formatUtilization(averageRentalUtilization)}
                                </h3>
                            }
                            theme={IslandTheme.GRAY}
                        >
                            {i18n('Avg. for the period')}
                        </Island>
                    </div>

                    {!isLoading ? (
                        <ChartViewHistogram
                            className={styles.chartHours}
                            points={rentalUtilizationChartPoints}
                            aggregationConfig={aggregationConfig}
                            keys={CHART_KEYS}
                            keysFormat={[
                                {
                                    fraction: 1,
                                    postfix: '%',
                                },
                            ]}
                        />
                    ) : (
                        <div className={styles.chartShimmer}>
                            <Spin />
                        </div>
                    )}
                </Island>

                <div className={styles.utilizationContainer}>
                    <TopTable
                        title={i18n('Top cars by utilisation')}
                        header={[i18n('Car'), '%']}
                        items={getTopTableItems(topRentalUtilizationCars, true)}
                        isLoading={isLoading}
                    />
                    <TopTable
                        title={i18n('Worst cars by utilisation')}
                        header={[i18n('Car'), '%']}
                        items={getTopTableItems(worseRentalUtilizationCars, true)}
                        isLoading={isLoading}
                    />
                </div>
            </div>
        </div>
    );
};
