import * as React from 'react';
import cn from 'classnames/bind';

import { CarBookingStatusType } from 'entities/Car/consts/CarBookingStatusType';

import { IslandTheme } from 'shared/consts/IslandTheme';
import { TextWithDotSize } from 'shared/consts/TextWithDotSize';
import { Island } from 'shared/ui/Island/Island';
import { Spin } from 'shared/ui/Spin/Spin';
import { TextWithDot } from 'shared/ui/TextWithDot/TextWithDot';

import { i18n } from 'pages/Dashboard/DashboardRental/ui/BookingStateBoard/BookingStateBoard.i18n';

import styles from 'pages/Dashboard/DashboardRental/ui/BookingStateBoard/BookingStateBoard.css';

export interface BookingStateBoardProps {
    className?: string;

    statuses?: Record<CarBookingStatusType, number>;
    totalCars?: number;
    isLoading: boolean;
}

const cx = cn.bind(styles);

const STATUS_ORDER: CarBookingStatusType[] = [
    CarBookingStatusType.IN_RENT,
    CarBookingStatusType.FREE,
    CarBookingStatusType.IN_SERVICE,
];
const STATUSES: Record<CarBookingStatusType, string> = {
    [CarBookingStatusType.IN_RENT]: i18n('In rent'),
    [CarBookingStatusType.IN_SERVICE]: i18n('In service'),
    [CarBookingStatusType.FREE]: i18n('Free'),
};

export const BookingStateBoard: React.FC<BookingStateBoardProps> = function BookingStateBoard({
    className,
    statuses,
    totalCars,
    isLoading,
}) {
    const [hoveredCarId, setHoveredCarId] = React.useState<Optional<string>>();
    const onMouseEnterItemHandler = React.useCallback(
        ({ currentTarget }: React.MouseEvent<HTMLDivElement>) => {
            const id = currentTarget.getAttribute('data-id') || '';

            if (hoveredCarId !== id) {
                setHoveredCarId(id);
            }
        },
        [hoveredCarId],
    );

    const onMouseLeaveHandler = React.useCallback(() => {
        setHoveredCarId(undefined);
    }, []);

    return (
        <>
            <Island
                title={
                    <div className={styles.islandTitleContainer}>
                        <h3 className={styles.islandTitle}>
                            {isLoading ? (
                                <Spin />
                            ) : (
                                i18n('{count} car', {
                                    count: totalCars,
                                })
                            )}
                        </h3>
                        <div className={styles.now}>{i18n('Now')}</div>
                    </div>
                }
            >
                <div className={styles.timeline}>
                    {totalCars && statuses ? (
                        STATUS_ORDER.map((status) => {
                            const data = statuses[status];

                            return data ? (
                                <div
                                    key={status}
                                    data-id={status}
                                    className={cx(styles.carStatusTimeline, { hovered: status === hoveredCarId }, [
                                        status,
                                    ])}
                                    onMouseEnter={onMouseEnterItemHandler}
                                    onMouseLeave={onMouseLeaveHandler}
                                    style={{
                                        flexBasis: `${(data / totalCars) * 100}%`,
                                    }}
                                />
                            ) : null;
                        })
                    ) : (
                        <div className={cx(styles.carStatusTimeline)} />
                    )}
                </div>

                <div className={styles.rowIslands}>
                    {STATUS_ORDER.map((status) => {
                        const data = statuses && statuses[status];

                        return (
                            <Island
                                key={status}
                                data-id={status}
                                className={cx(styles.islandNumber, { hovered: status === hoveredCarId })}
                                onMouseEnterItem={onMouseEnterItemHandler}
                                onMouseLeaveItem={onMouseLeaveHandler}
                                title={
                                    data ? (
                                        <TextWithDot
                                            className={cx(styles.carStatus, [status])}
                                            size={TextWithDotSize.L}
                                            reverse
                                        >
                                            {data}
                                        </TextWithDot>
                                    ) : (
                                        <div className={cx(styles.carStatusSpin)}>
                                            <Spin />
                                        </div>
                                    )
                                }
                                theme={IslandTheme.GRAY}
                            >
                                {STATUSES[status]}
                            </Island>
                        );
                    })}
                </div>
            </Island>
        </>
    );
};
