import * as React from 'react';
import { useHistory, useLocation } from 'react-router-dom';

import { ONE_SECOND, ONE_WEEK } from 'constants/constants';

import { DashboardFilter } from 'pages/Dashboard/Dashboard/conts/DashboardFilter';
import { EfficiencyBoard } from 'pages/Dashboard/Dashboard/ui/EfficiencyBoard/EfficiencyBoard';
import { StateBoard } from 'pages/Dashboard/Dashboard/ui/StateBoard/StateBoard';
import { BookingEfficiencyBoard } from 'pages/Dashboard/DashboardRental/ui/BookingEfficiencyBoard/BookingEfficiencyBoard';
import { RevenueBoard } from 'pages/Dashboard/DashboardRental/ui/RevenueBoard/RevenueBoard';

import { Header } from 'widgets/Header';

import { FilterDate } from 'features/FilterDate/ui/FilterDate/FilterDate';

import { CarNotFound } from 'entities/Car/ui/CarNotFound/CarNotFound';
import { useRidesSummary, UseRidesSummaryTariff } from 'entities/Ride/api/useRidesSummary/useRidesSummary';

import { DateFilter } from 'shared/consts/DateFilter';
import { getDateAggregation } from 'shared/helpers/getDateAggregation/getDateAggregation';
import { getDateEndDay } from 'shared/helpers/getDateEndDay/getDateEndDay';
import { getDateStartDay } from 'shared/helpers/getDateStartDay/getDateStartDay';
import { updateFiltersParams } from 'shared/helpers/updateFiltersParams/updateFiltersParams';
import { useRetry } from 'shared/hooks/useRetry/useRetry';
import { useSearchFilters } from 'shared/hooks/useSearchFilters/useSearchFilters';
import { ContentContainer } from 'shared/ui/ContentContainer/ContentContainer';
import { ErrorBoundary } from 'shared/ui/ErrorBoundary/ErrorBoundary';
import { PageLoading } from 'shared/ui/PageLoading/PageLoading';
import { SectionLayout } from 'shared/ui/SectionLayout/SectionLayout';

import { RidesFiltersOptions } from 'components/Rides';

import { i18n } from 'pages/Dashboard/DashboardRental/ui/DashboardRental/DashboardRental.i18n';

import styles from 'pages/Dashboard/DashboardRental/ui/DashboardRental/DashboardRental.css';

export interface DashboardRentalProps {
    className?: string;
}

export const DashboardRental: React.FC<DashboardRentalProps> = function DashboardRental({ className }) {
    const location = useLocation();
    const history = useHistory();

    // @todo: moved into component because storybook set mock date only for one story; set mock date for all app
    const initialDashboardFilters = {
        [DateFilter.SINCE]: (getDateStartDay(new Date()).getTime() - ONE_WEEK * 2) / ONE_SECOND,
        [DateFilter.UNTIL]: Math.trunc(getDateEndDay(new Date()).getTime() / ONE_SECOND),
    };
    const existingDashboardFilters = [DateFilter.SINCE, DateFilter.UNTIL];
    const dateDashboardFilters = [DateFilter.SINCE, DateFilter.UNTIL];

    const filters = useSearchFilters<RidesFiltersOptions>(initialDashboardFilters, existingDashboardFilters, {
        dateFilters: dateDashboardFilters,
    });

    const aggregationConfig = getDateAggregation({
        since: filters[DateFilter.SINCE],
        until: filters[DateFilter.UNTIL],
    });

    const {
        data: summary,
        isLoading: isSummaryLoading,
        error: summaryError,
        reload: summaryReload,
    } = useRidesSummary({
        since: filters[DateFilter.SINCE],
        until: filters[DateFilter.UNTIL],
        aggregation: aggregationConfig.aggregation,
        tariff: UseRidesSummaryTariff.RENTAL_OFFER,
    });
    useRetry(summaryError, summaryReload);

    const {
        data: standartSummary,
        isLoading: isStandartSummaryLoading,
        error: standartSummaryError,
        reload: standartSummaryReload,
    } = useRidesSummary({
        since: filters[DateFilter.SINCE],
        until: filters[DateFilter.UNTIL],
        aggregation: aggregationConfig.aggregation,
        tariff: undefined,
    });
    useRetry(standartSummaryError, standartSummaryReload);

    const { totalCars, statuses } = standartSummary || {};

    const onChangeFilterHandler = React.useCallback(
        (type: string, value: Record<DateFilter, Nullable<number>>) => {
            const searchParams = new URLSearchParams(location?.search);

            history.push(`${location.pathname}?${updateFiltersParams(searchParams, type, value)}`);
        },
        [location],
    );

    return (
        <ErrorBoundary fallback={CarNotFound}>
            <SectionLayout
                header={
                    <Header
                        title={i18n('Dashboard')}
                        withoutBorder
                    />
                }
                bodyScroll
            >
                <React.Suspense fallback={<PageLoading />}>
                    <ContentContainer bodyScroll>
                        <>
                            <div>
                                <div className={styles.filters}>
                                    <h2 className={styles.rowTitle}>{i18n('Realtime states')}</h2>

                                    <FilterDate
                                        className={styles.filterDate}
                                        type={DashboardFilter.DATE}
                                        filters={filters}
                                        onChangeFilter={onChangeFilterHandler}
                                        placement="bottom-end"
                                        inline
                                    />
                                </div>

                                <StateBoard
                                    statuses={statuses}
                                    totalCars={totalCars}
                                    isLoading={isStandartSummaryLoading}
                                />

                                {/* DRIVEMATICSDEV-617
                                    <BookingStateBoard
                                        statuses={statuses}
                                        totalCars={totalCars}
                                        isLoading={isLoading}
                                    />
                                 */}
                            </div>

                            <div className={styles.row}>
                                <h2 className={styles.rowTitle}>{i18n('Revenue')}</h2>

                                <RevenueBoard
                                    isLoading={isSummaryLoading}
                                    summary={summary}
                                    aggregationConfig={aggregationConfig}
                                />
                            </div>

                            <div className={styles.row}>
                                <h2 className={styles.rowTitle}>{i18n('Bookings')}</h2>

                                <BookingEfficiencyBoard
                                    isLoading={isSummaryLoading}
                                    summary={summary}
                                    aggregationConfig={aggregationConfig}
                                />
                            </div>

                            <div className={styles.row}>
                                <h2 className={styles.rowTitle}>{i18n('Fleet utilization')}</h2>

                                <EfficiencyBoard
                                    isLoading={isStandartSummaryLoading}
                                    summary={standartSummary}
                                    aggregationConfig={aggregationConfig}
                                />
                            </div>
                        </>
                    </ContentContainer>
                </React.Suspense>
            </SectionLayout>
        </ErrorBoundary>
    );
};
