import * as React from 'react';
import cn from 'classnames';

import { formatCarNumber } from 'entities/Car';
import {
    CarTop,
    UseRidesSummaryAddonSale,
    UseRidesSummaryRes,
} from 'entities/Ride/api/useRidesSummary/useRidesSummary';

import { IslandTheme } from 'shared/consts/IslandTheme';
import { Path } from 'shared/consts/Path';
import { formatCurrency } from 'shared/helpers/formatCurrency/formatCurrency';
import { generateRouterPath } from 'shared/helpers/generateRouterPath/generateRouterPath';
import { DateAggregationConfig } from 'shared/helpers/getDateAggregation/getDateAggregation';
import { getDateAggregationLabel } from 'shared/helpers/getDateAggregationLabel/getDateAggregationLabel';
import { ChartKey } from 'shared/types/ChartPoint';
import { ChartViewHistogram } from 'shared/ui/ChartViewHistogram/ChartViewHistogram';
import { Island } from 'shared/ui/Island/Island';
import { Link } from 'shared/ui/Link/Link';
import { Spin } from 'shared/ui/Spin/Spin';
import { TopTable, TopTableItem } from 'shared/ui/TopTable/TopTable';

import { i18n } from 'pages/Dashboard/DashboardRental/ui/RevenueBoard/RevenueBoard.i18n';

import styles from 'pages/Dashboard/DashboardRental/ui/RevenueBoard/RevenueBoard.css';

export interface RevenueBoardProps {
    className?: string;

    summary?: UseRidesSummaryRes;
    isLoading: boolean;
    aggregationConfig: DateAggregationConfig;
}

const CHART_KEYS: ChartKey[] = ['line1'];

function getTopSalesTableItems(top: Optional<UseRidesSummaryAddonSale[]>): TopTableItem[] {
    if (!top) {
        return [];
    }

    return top.map((addon, index): TopTableItem => {
        const value = addon.count;

        if (!addon.id) {
            return {
                id: index,
                text: null,
                value,
            };
        }

        return {
            id: addon.id,
            text: <div className={styles.addonName}>{addon.title}</div>,
            value,
        };
    });
}

function getTopTableItems(top: Optional<CarTop[]>, currency?: string): TopTableItem[] {
    if (!top) {
        return [];
    }

    return top.map((car, index): TopTableItem => {
        return {
            id: car.object_id,
            text: (
                <Link
                    href={generateRouterPath(Path.CAR, { id: car.object_id })}
                    className={styles.car}
                >
                    <div className={styles.carName}>{car.name}</div>
                    <div className={styles.carNumber}>{formatCarNumber(car.number)}</div>
                </Link>
            ),
            value: formatCurrency(car.value, currency),
        };
    });
}

export const RevenueBoard: React.FC<RevenueBoardProps> = function RevenueBoard({
    className,
    aggregationConfig,
    summary,
    isLoading,
}) {
    const AveragePerTime = getDateAggregationLabel(aggregationConfig.aggregation);

    const {
        currency,
        averageCheck,
        revenueAvgPerTime,
        revenueTotalForThePeriod,
        revenueChartPoints = [],
        averageCheckChartPoints = [],
        addonSales,
        insuranceSales,
        topRevenueCars,
        worseRevenueCars,
        topAverageCheckCars,
        worseAverageCheckCars,
    } = summary || {};

    return (
        <div className={cn(styles.container, className)}>
            <div className={styles.column}>
                <Island title={i18n('Revenue')}>
                    <div className={styles.tabletRow}>
                        <Island
                            className={styles.tablet}
                            title={
                                <h3 className={styles.islandSubTitle}>
                                    {isLoading ? <Spin /> : formatCurrency(revenueTotalForThePeriod, currency)}
                                </h3>
                            }
                            theme={IslandTheme.GRAY}
                        >
                            {i18n('Total for the period')}
                        </Island>

                        <Island
                            className={styles.tablet}
                            title={
                                <h3 className={styles.islandSubTitle}>
                                    {isLoading ? <Spin /> : formatCurrency(revenueAvgPerTime, currency)}
                                </h3>
                            }
                            theme={IslandTheme.GRAY}
                        >
                            {AveragePerTime}
                        </Island>
                    </div>

                    {!isLoading ? (
                        <ChartViewHistogram
                            className={styles.chartRevenue}
                            points={revenueChartPoints}
                            aggregationConfig={aggregationConfig}
                            keys={CHART_KEYS}
                            keysFormat={[
                                {
                                    fraction: 0,
                                },
                            ]}
                        />
                    ) : (
                        <div className={styles.chartShimmer}>
                            <Spin />
                        </div>
                    )}
                </Island>

                <div className={styles.check}>
                    <TopTable
                        title={i18n('Top cars by revenue')}
                        header={[i18n('Car'), currency]}
                        items={getTopTableItems(topRevenueCars, currency)}
                        isLoading={isLoading}
                    />
                    <TopTable
                        title={i18n('Worst cars by revenue')}
                        header={[i18n('Car'), currency]}
                        items={getTopTableItems(worseRevenueCars, currency)}
                        isLoading={isLoading}
                    />
                </div>

                <div className={styles.sales}>
                    <TopTable
                        title={i18n('Addons sales')}
                        items={getTopSalesTableItems(addonSales)}
                        isLoading={isLoading}
                    />
                    <TopTable
                        title={i18n('Insurance sales')}
                        items={getTopSalesTableItems(insuranceSales)}
                        isLoading={isLoading}
                    />
                </div>
            </div>

            <div className={styles.column}>
                <Island title={i18n('Average check')}>
                    <div className={styles.tabletRow}>
                        <Island
                            className={styles.tablet}
                            title={
                                <h3 className={styles.islandSubTitle}>
                                    {isLoading ? <Spin /> : formatCurrency(averageCheck, currency)}
                                </h3>
                            }
                            theme={IslandTheme.GRAY}
                        >
                            {i18n('Avg. for the period')}
                        </Island>
                    </div>

                    {!isLoading ? (
                        <ChartViewHistogram
                            className={styles.chartCheck}
                            points={averageCheckChartPoints}
                            aggregationConfig={aggregationConfig}
                            keys={CHART_KEYS}
                            keysFormat={[
                                {
                                    fraction: 0,
                                },
                            ]}
                        />
                    ) : (
                        <div className={styles.chartShimmer}>
                            <Spin />
                        </div>
                    )}
                </Island>

                <div className={styles.check}>
                    <TopTable
                        title={i18n('Top cars by average check')}
                        header={[i18n('Car'), currency]}
                        items={getTopTableItems(topAverageCheckCars, currency)}
                        isLoading={isLoading}
                    />
                    <TopTable
                        title={i18n('Worst cars by average check')}
                        header={[i18n('Car'), currency]}
                        items={getTopTableItems(worseAverageCheckCars, currency)}
                        isLoading={isLoading}
                    />
                </div>
            </div>
        </div>
    );
};
