import * as React from 'react';
import { useState } from 'react';

import { ONE_SECOND, ONE_WEEK } from 'constants/constants';

import { Header, HEADER_HEIGHT } from 'widgets/Header';
import { SignalsContentHeader } from 'widgets/SignalsContentHeader';
import { SignalsFilters } from 'widgets/SignalsFilters';
import { SignalsTableWidget } from 'widgets/SignalsTableWidget';

import { SignalsHeaderTabs } from 'features/SignalsHeaderTabs';

import { useSignalList } from 'entities/Signal/api/useSignalList/useSignalList';
import { INITIAL_SIGNALS_FILTERS } from 'entities/Signal/consts/filters';
import { SignalFormattedSchema } from 'entities/Signal/types/SignalFormattedSchema';

import { DateFilter } from 'shared/consts/DateFilter';
import { getDateStartDay } from 'shared/helpers/getDateStartDay/getDateStartDay';
import { ContentContainer } from 'shared/ui/ContentContainer/ContentContainer';
import { SectionLayout } from 'shared/ui/SectionLayout/SectionLayout';
import { Widget } from 'shared/ui/Widget/Widget';

import { SignalsFiltersOptions } from 'components/Signals';

import { i18n } from 'pages/Signals/Signals/ui/SignalsPage/SignalsPage.i18n';

import styles from 'pages/Signals/Signals/ui/SignalsPage/SignalsPage.css';

export interface SignalsPageProps {
    className?: string;
}

const INITIAL_FILTERS = {
    ...INITIAL_SIGNALS_FILTERS,
    [DateFilter.SINCE]: (getDateStartDay(new Date()).getTime() - ONE_WEEK) / ONE_SECOND,
};

export const SignalsPage: React.FC<SignalsPageProps> = function SignalsPage({ className }) {
    const [filters, setFilters] = useState<SignalsFiltersOptions>(INITIAL_FILTERS);
    const { data, isFullyLoaded, isLoading, fetchNextPage, updateData } = useSignalList(filters);

    const onRemoveData = React.useCallback(
        (tag_id: string) => {
            updateData((data) => {
                return data && data.filter((item) => item.tag_id !== tag_id);
            });
        },
        [updateData],
    );
    const onUpdateData = React.useCallback(
        (tag_id: string, newData: Partial<SignalFormattedSchema>) => {
            updateData((data) => {
                return (
                    data &&
                    data.map((item) => {
                        if (item.tag_id === tag_id) {
                            return {
                                ...item,
                                ...newData,
                            };
                        }

                        return item;
                    })
                );
            });
        },
        [updateData],
    );

    return (
        <SectionLayout
            header={
                <Header
                    title={i18n('Signals')}
                    tabs={<SignalsHeaderTabs />}
                    withoutBorder
                />
            }
            filters={
                <SignalsFilters
                    offsetTop={HEADER_HEIGHT}
                    initialFilters={INITIAL_FILTERS}
                    onFiltersChange={setFilters}
                />
            }
            bodyScroll
        >
            <ContentContainer
                className={styles.contentContainer}
                bodyScroll
                withSidebar
            >
                <Widget contentContainer>
                    <SignalsContentHeader filters={filters} />

                    <SignalsTableWidget
                        data={data}
                        isFullyLoaded={isFullyLoaded}
                        isLoading={isLoading}
                        fetchNextPage={fetchNextPage}
                        onRemoveData={onRemoveData}
                        onUpdateData={onUpdateData}
                    />
                </Widget>
            </ContentContainer>
        </SectionLayout>
    );
};
