import * as React from 'react';
import { useParams } from 'react-router-dom';

import getFinish from 'utils/sessions/getFinish';
import { ISession } from 'utils/sessions/types';

import { Header, HEADER_HEIGHT } from 'widgets/Header';
import { RidesContentHeader } from 'widgets/RidesContentHeader';
import { RidesFilters } from 'widgets/RidesFilters';

import { UserHeaderTabs } from 'features/UserHeaderTabs';

import { INITIAL_RIDES_FILTERS } from 'entities/Ride/consts/filters';
import { UserHeaderTitle, UserNotFound, useUserMember } from 'entities/User';

import { ContentContainer } from 'shared/ui/ContentContainer/ContentContainer';
import { ErrorBoundary } from 'shared/ui/ErrorBoundary/ErrorBoundary';
import { PageLoading } from 'shared/ui/PageLoading/PageLoading';
import { SectionLayout } from 'shared/ui/SectionLayout/SectionLayout';
import { Widget } from 'shared/ui/Widget/Widget';

import { RideFiltersType, RidesFiltersOptions } from 'components/Rides';
import { REQUESTS, RIDES_REQUESTS } from 'components/Rides/request';
import { RidesTable } from 'components/Rides/RidesTable';

import { RequestHelper } from './../../../../../../request-helper/src';

// @todo: add storybook test
export const UserRidesPage: React.FC = function UserRidesPage() {
    const { id } = useParams<{ id: string }>();

    const userMemberResource = useUserMember(id);

    // @todo: rewrite
    const request: React.MutableRefObject<RequestHelper> = React.useRef(
        new RequestHelper({ requestConfigs: RIDES_REQUESTS }),
    );
    const [columns, setColumns] = React.useState<string[]>([]);
    const [filters, setFilters] = React.useState<RidesFiltersOptions>(INITIAL_RIDES_FILTERS);

    const fetchRides = React.useCallback(
        (options: Partial<RideFiltersType>) => {
            return request.current
                .exec(REQUESTS.GET_RIDES, {
                    queryParams: {
                        ...options,
                        user_id: id,
                    },
                })
                .then((response) => {
                    return response;
                })
                .catch((error) => {
                    throw error;
                });
        },
        [id],
    );

    const getRides = React.useCallback(async (since: number | null, until: number | null) => {
        return fetchRides({
            since,
            until,
            numdoc: 50,
        })
            .then((response: { has_more: boolean; sessions: ISession[]; models: any }) => {
                const lastSession = response.sessions?.[response.sessions?.length - 1];

                return {
                    models: response.models,
                    sessions: response.sessions,
                    canGetMore: response.has_more,
                    currentUntil: response.has_more ? getFinish(lastSession)?.timestamp ?? null : null,
                };
            })
            .catch((error) => {
                throw error;
            });
    }, []);

    const onTableColumnsChangeHandler = React.useCallback((columns: string[]) => {
        setColumns(columns);
    }, []);

    return (
        <ErrorBoundary fallback={<UserNotFound />}>
            <React.Suspense fallback={<PageLoading />}>
                <SectionLayout
                    header={
                        <Header
                            title={<UserHeaderTitle resource={userMemberResource} />}
                            tabs={<UserHeaderTabs resource={userMemberResource} />}
                            withoutBorder
                        />
                    }
                    filters={
                        <RidesFilters
                            offsetTop={HEADER_HEIGHT}
                            onFiltersChange={setFilters}
                        />
                    }
                    bodyScroll
                >
                    <ContentContainer
                        bodyScroll
                        withSidebar
                    >
                        <Widget contentContainer>
                            <RidesContentHeader
                                filters={filters}
                                getData={fetchRides}
                                hasDriverColumn={false}
                                onCarsTableColumnsChange={onTableColumnsChangeHandler}
                            />

                            <RidesTable
                                filters={filters}
                                columns={columns}
                                hasDriverColumn={false}
                                getRides={getRides}
                            />
                        </Widget>
                    </ContentContainer>
                </SectionLayout>
            </React.Suspense>
        </ErrorBoundary>
    );
};
