import { axisBottom } from 'd3-axis';
import { ScaleTime } from 'd3-scale';
import { select as d3Select } from 'd3-selection';

import { CHART_AXIS_MARGIN_BOTTOM, CHART_MARGIN_TOP } from 'shared/consts/Chart';
import { formatDateToString } from 'shared/helpers/formatDateToString/formatDateToString';
import { isDateWeekend } from 'shared/helpers/isDateWeekend/isDateWeekend';
import { ChartPoint } from 'shared/types/ChartPoint';
import { D3SelectFunction } from 'shared/types/D3SelectFunction';

const TICK_PADDING = 10;
const MAX_DATE = 14;
const SPARSITY_RATIO = 7;

export function d3SelectDays(
    points: ChartPoint[],
    scaleTime: ScaleTime<number, number>,
    height: number,
    element: Nullable<SVGGElement>,
): void {
    const isSparsity = MAX_DATE < points.length;
    const sparsity = isSparsity ? Math.floor(points.length / SPARSITY_RATIO) : 0;
    const lastIndex = points.length - 1;
    let markedDates = points.map(({ begin_date: date }) => new Date(date));
    let xAxis = axisBottom<Date>(scaleTime)
        .tickSize(height - CHART_MARGIN_TOP - CHART_AXIS_MARGIN_BOTTOM)
        .tickPadding(TICK_PADDING)
        .tickValues(markedDates)
        .tickFormat((date, index) => {
            const shouldSkipBeforeLast = isSparsity && index + sparsity > lastIndex;
            const shouldSkip = sparsity > 0 && (index % sparsity !== 0 || shouldSkipBeforeLast);

            if (shouldSkip && index !== lastIndex) {
                return '';
            }

            return formatDateToString(date, true)!;
        });

    d3Select(element)
        .call(xAxis as D3SelectFunction)
        .call((g) => g.selectAll('.tick line').remove())
        .call((g) =>
            g
                .selectAll('.tick text')
                .filter((d) => {
                    return isDateWeekend(d as Date);
                })
                .attr('class', 'weekend'),
        );
}
