import '@yandex-int/error-counter/dist/interfaceWithoutRum';
import '@yandex-int/error-counter/dist/implementation';
import '@yandex-int/error-counter/dist/filters';
import '@yandex-int/error-counter/dist/sendWithoutRum';
import '@yandex-int/error-counter/dist/logError';
import '@yandex-int/error-counter/dist/logAjaxError';

export enum ErrorLoggerSource {
    NETWORK = 'network',
    ERROR_BOUNDARY = 'error-boundary',
}

function getFrontendEnv(): 'production' | 'testing' | 'prestable' | 'development' {
    const origin = location.origin;

    if (/^https:\/\/(.+\.)?vlootkit\.(com|cz)$/.test(origin)) {
        return 'production';
    }

    if (/^https:\/\/drivematics\.yandex\.(ru|com|eu)$/.test(origin)) {
        return 'production';
    }

    if (/^https:\/\/(.+\.)?testing\.drivematics\.yandex\.(ru|com|eu)$/.test(origin)) {
        return 'testing';
    }

    if (/^https:\/\/(.+\.)?prestable\.drivematics\.yandex\.(ru|com|eu)$/.test(origin)) {
        return 'prestable';
    }

    return 'development';
}

export function initErrorLogger(uid?: string) {
    const env = getFrontendEnv();

    if (env !== 'development') {
        window.Ya.Rum.initErrors({
            project: 'drivematics',
            reqid: `${new Date().toISOString()}-${uid ?? 'unknown'}`,
            slots: [],

            /* optional */
            env,
            version: process.env.DM_VERSION,
            referrer: true,
            preventError: false,
            uncaughtException: true,
            debug: true,
            limits: {
                client: 1000,
            },
            silent: {
                external: 3,
            },
            transform: function (error) {
                if (!error.stack) {
                    error.additional = {
                        text: 'No stacktrace',
                    };
                }

                if (error.message) {
                    let msg = error.message.toLowerCase();

                    if (msg.includes('unhandled rejection: loading chunk')) {
                        error.additional = {
                            text: error.message,
                        };

                        error.message = 'Loading chunk failed';
                    }

                    if (msg.includes('unhandled rejection: loading css chunk')) {
                        error.additional = {
                            text: error.message,
                        };

                        error.message = 'Loading CSS chunk failed';
                    }
                }

                return error;
            },
        });
    }
}

export function logError(source: ErrorLoggerSource, error: Error, info?: Record<string, any>) {
    if (window.Ya?.Rum?.logError) {
        window.Ya.Rum.logError(
            {
                source,
                additional: info,
                page: location.href,
            },
            error,
        );
    }
}
