import { HttpStatusCode } from 'shared/consts/HttpStatusCode';
import { ErrorLoggerSource, logError } from 'shared/helpers/errorLogger/errorLogger';
import { isVlootkit } from 'shared/helpers/isVlootkit/isVlootkit';
import { CacheRequestContext } from 'shared/hooks/useCacheRequestContext/useCacheRequestContext';

// const VERSION = process.env.DM_VERSION || '';
const REQID_MASK = 200000000000;
const DEFAULT_TIMEOUT = 5000;

export interface FetchRequestOptions<S, R> {
    method?: 'GET' | 'POST' | 'PUT' | 'DELETE';
    timeout?: number;
    query?: Dict;
    cache?: CacheRequestContext;
    headers?: Dict;

    parser?(res: S): R;
}

export interface FetchRequestResult<T> {
    then(success: (data: T) => any, error?: (error: any) => any);

    transform<R>(transformer: (data: T) => R, error?: (error: any) => any): FetchRequestResult<R>;

    ready(): Promise<T>;

    cancel(): void;
}

export interface FetchErrorPayload {
    status: number;
    statusText: string;
    req: {
        url: string;
        query;
    };
    res: any;
    time: number;
    canceled: boolean;
}

export interface FetchErrorSerialized {
    __FetchError__: {
        message: string;
        payload: FetchErrorPayload;
    };
}

export class FetchError extends Error implements FetchErrorPayload {
    public name: string;
    public message: string;
    public status: number;
    public statusText: string;
    public req: {
        url: string;
        query: string;
    };
    public res: any;
    public time: number;
    public canceled: boolean;

    constructor(message: string, { status, statusText, req, res, time, canceled }: FetchErrorPayload) {
        super(message);
        this.status = status;
        this.statusText = statusText;
        this.req = req;
        this.res = res;
        this.time = time;
        this.canceled = canceled;
        Object.setPrototypeOf(this, FetchError.prototype);
    }

    serialize(): FetchErrorSerialized {
        return {
            __FetchError__: {
                message: this.message,
                payload: {
                    status: this.status,
                    statusText: this.statusText,
                    req: this.req,
                    res: this.res,
                    time: this.time,
                    canceled: this.canceled,
                },
            },
        };
    }

    static factory(error: FetchErrorSerialized): FetchError {
        const data = error.__FetchError__;

        return new FetchError(data.message, data.payload);
    }
}

function isSerializedFetchError(error: any): error is FetchErrorSerialized {
    return Boolean(error && error.__FetchError__);
}

function isFetchPendingPromise(error: any): boolean {
    return Boolean(error && error.__FetchPendingPromise__);
}

function _noop() {}

function _createAbortController() {
    try {
        if (typeof window !== 'undefined' && window.AbortController) {
            return new AbortController();
        }
    } catch (e) {
        // AbortController not supported
    }

    return {
        signal: undefined,
        abort() {
            // noop
        },
    };
}

function _setTimeout(cb: () => void, timeout: number): () => void {
    let timeoutId: number | undefined = window.setTimeout(cb, timeout);

    return function cleanupTimeout() {
        clearTimeout(timeoutId);
        timeoutId = undefined;
    };
}

async function _fetch<R>(url: string, query: URLSearchParams, options: RequestInit): Promise<R> {
    const queryStr = query.toString();

    const startTime = Date.now();

    try {
        const response = await fetch(url + (queryStr.length ? '?' + queryStr : ''), options);

        if (response.status === HttpStatusCode.UNAUTHORIZED && isVlootkit() && !query.get('__retry__')) {
            await fetch('/auth/api/refresh', {
                method: 'GET',
                credentials: 'same-origin',
            });

            const cloneQuery = new URLSearchParams(query.toString());
            cloneQuery.set('__retry__', '1');

            return _fetch(url, cloneQuery, options);
        }

        const contentType = response.headers.get('content-type');

        if (contentType?.startsWith('image/')) {
            return response.blob() as unknown as Promise<R>;
        }

        let hasParseError = false;

        let responseText = await response.text();
        let responseData: R;

        const responseTime = Date.now() - startTime;
        const isJsonResponse = contentType?.startsWith('application/json');

        if (response.ok && ((options.method !== 'GET' && !responseText) || !isJsonResponse)) {
            return responseText as unknown as R;
        }

        try {
            responseData = JSON.parse(responseText);
        } catch (error) {
            hasParseError = true;
        }

        if (response.ok && !hasParseError) {
            return responseData!;
        } else {
            const errorData = hasParseError ? responseText : responseData!;

            logError(
                ErrorLoggerSource.NETWORK,
                new Error(
                    JSON.stringify({
                        url,
                        status: response.status,
                    }),
                ),
                {
                    response: errorData,
                },
            );

            throw new FetchError(hasParseError ? 'Fetch response parse error' : 'Fetch response error', {
                status: response.status,
                statusText: response.statusText,
                req: {
                    url,
                    query: queryStr,
                },
                res: errorData,
                time: responseTime,
                canceled: false,
            });
        }
    } catch (error) {
        if (error instanceof FetchError) {
            // pipe FetchError
            throw error;
        }

        if (error && (error.message === 'AbortError' || (error.ABORT_ERR && error.code === error.ABORT_ERR))) {
            // canceled
            throw new FetchError('Fetch request canceled error', {
                status: 0,
                statusText: 'Request Canceled',
                req: {
                    url,
                    query: queryStr,
                },
                res: undefined,
                time: Date.now() - startTime,
                canceled: true,
            });
        }

        // unexpected behaviour
        throw error;
    }
}

export function buildFetchRequestResult<T>(promise: Promise<T>, cancel: () => void = _noop): FetchRequestResult<T> {
    return {
        // await method
        then(success, error) {
            return promise.then(success, error);
        },
        transform<R>(transformer: (data: T) => R, error): FetchRequestResult<R> {
            return buildFetchRequestResult(promise.then(transformer, error), cancel);
        },
        ready() {
            return promise;
        },
        cancel,
    };
}

const LANG = process.env.LANG || 'en';

function normalizeParams(obj: Optional<Record<string, any>>): Dict {
    if (!obj) {
        return {};
    }

    return Object.keys(obj).reduce((memo, key) => {
        if (obj[key] !== null && obj[key] !== undefined) {
            memo[key] = String(obj[key]);
        }

        return memo;
    }, {});
}

export function fetchRequest<D extends Record<string, any>, R, S = R>(
    url: string,
    data: D,
    options: FetchRequestOptions<S, R> = {},
): FetchRequestResult<R> {
    const { method = 'GET', timeout = DEFAULT_TIMEOUT, cache, parser } = options;

    const query = new URLSearchParams(normalizeParams(method === 'GET' ? data : options.query));
    const isFile = data instanceof File;

    query.append('reqid', `${new Date().getTime()}-${(Math.random() * REQID_MASK).toFixed(0)}-DM`);

    if (!query.get('lang')) {
        query.append('lang', LANG);
    }

    const headers: Dict = {
        // 'X-DM-VERSION': VERSION,
        ...options.headers,
    };

    if (method !== 'GET' && !isFile) {
        headers['Content-Type'] = 'application/json';
    }

    const abortController = _createAbortController();

    const cleanupTimeout = _setTimeout(() => abortController.abort(), timeout);

    function buildCacheKey(): string {
        return JSON.stringify({
            method,
            url,
            data,
            query: options.query,
        });
    }

    let promise: Optional<Promise<S>>;

    if (process.env.DM_CACHE_REQUEST) {
        // CACHE: readonly, write
        const cacheKey = buildCacheKey();

        if (cache && cache.has(cacheKey)) {
            const value = cache.get(cacheKey);

            if (isFetchPendingPromise(value)) {
                promise = new Promise<S>(() => undefined);
            } else if (isSerializedFetchError(value)) {
                promise = Promise.reject(FetchError.factory(value));
            } else {
                promise = Promise.resolve(value as S);
            }
        }

        if (!cache) {
            throw new Error(`fetchRequest cache(CacheRequestContext) not found for ${cacheKey}`);
        }
    }

    if (!promise) {
        if (process.env.DM_CACHE_REQUEST === 'readonly') {
            // CACHE: readonly
            throw new Error(`fetchRequest cache not found. [url: '${url}', data: ${JSON.stringify(data)}]`);
        }

        promise = _fetch<S>(url, query, {
            method,
            signal: abortController.signal,
            headers,
            referrerPolicy: 'origin-when-cross-origin',
            credentials: 'same-origin',
            body: isFile ? (data as any) : method !== 'GET' ? JSON.stringify(data) : null,
        });

        if (process.env.DM_CACHE_REQUEST === 'write') {
            // CACHE: write
            const cacheKey = buildCacheKey();

            if (cache) {
                promise.then(
                    (data) => {
                        cache!.set(cacheKey, data);
                    },
                    (error) => {
                        if (error instanceof FetchError) {
                            cache!.set(cacheKey, error.serialize());
                        }
                    },
                );
            }
        }
    }

    const parsePromise = promise.then((result) => {
        if (parser) {
            return parser(result);
        }

        return result as unknown as R;
    });

    parsePromise.then(cleanupTimeout, cleanupTimeout);

    function cancel() {
        cleanupTimeout();
        abortController.abort();
    }

    return buildFetchRequestResult(parsePromise, cancel);
}
