import * as React from 'react';

import { MenuType } from 'shared/consts/MenuType';

function convertToObj(value: Optional<string | string[]>): Dict<boolean> {
    if (!value) {
        return {};
    }

    if (typeof value === 'string') {
        return { [value]: true };
    }

    return value.reduce((memo: Dict<boolean>, item) => {
        memo[item] = true;

        return memo;
    }, {});
}

function convertToValue(value: Dict<boolean>): string | string[] {
    const values = Object.keys(value).filter((key) => value[key]);

    return values.length > 1 ? values : values[0];
}

export function useCheckedMenu(
    value?: string | string[],
    menuType?: MenuType,
    onClick?: (event: React.MouseEvent<HTMLLIElement>) => void,
): { checked: string | string[]; onMenuItemClickHandler: (event: React.MouseEvent<HTMLLIElement>) => void } {
    const [valueObj, setValueObj] = React.useState<Dict<boolean>>(convertToObj(value));

    React.useEffect(() => {
        setValueObj(convertToObj(value));
    }, [value]);

    const onMenuItemClickHandler = React.useCallback(
        (event: React.MouseEvent<HTMLLIElement>) => {
            if (menuType) {
                const target = event.currentTarget;
                const currentValue = target.getAttribute('data-value');

                if (currentValue) {
                    if (menuType === MenuType.CHECK) {
                        let values = { ...valueObj };
                        let value = values[currentValue];

                        values[currentValue] = !value;

                        setValueObj(values);
                    }

                    if (menuType === MenuType.RADIO) {
                        if (!valueObj[currentValue]) {
                            setValueObj({ [currentValue]: true });
                        }
                    }
                }
            }

            if (onClick) {
                onClick(event);
            }
        },
        [menuType, onClick, valueObj],
    );

    return { checked: convertToValue(valueObj), onMenuItemClickHandler };
}
