import { useCallback, useEffect, useState } from 'react';

import { FetchError, fetchRequest, FetchRequestOptions } from 'shared/helpers/fetchRequest/fetchRequest';
import { useCacheRequestContext } from 'shared/hooks/useCacheRequestContext/useCacheRequestContext';

export interface UseFetchOptions<S, R> extends FetchRequestOptions<S, R> {
    ttl?: number;
}

export interface UseFetchResource<R> {
    getKey: () => string;
    read: () => R;
    reload: (force?: boolean) => void;
}

interface FetchCacheItem<R> {
    promise?: Promise<void>;
    error?: Optional<FetchError | Error>;
    res?: R;

    pub: UseFetchResource<R>;
}

const cache: Map<string, FetchCacheItem<any>> = new Map();

export function useFetch<D, R, S = R>(
    url: string,
    data: D,
    options?: UseFetchOptions<S, R>,
    disabled?: boolean,
): UseFetchResource<R> {
    const cacheContext = useCacheRequestContext();

    // eslint-disable-next-line @typescript-eslint/no-unused-vars
    const [_, setVer] = useState<number>(0);

    const cacheKey = JSON.stringify({ url, data, options, key: cacheContext.getCacheName() });

    useEffect(() => {
        return () => {
            // cleanup cache old cache (params change or parent component destruct)
            if (cache.has(cacheKey)) {
                cache.delete(cacheKey);
            }
        };
    }, [cacheKey]);

    const rerender = useCallback(() => {
        setVer((ver) => ver + 1);
    }, [setVer]);

    if (cache.has(cacheKey)) {
        return cache.get(cacheKey)!.pub;
    }

    const cacheItem: FetchCacheItem<R> = {
        pub: {
            getKey() {
                return cacheKey;
            },
            read() {
                // success
                if (cacheItem.hasOwnProperty('res')) {
                    return cacheItem.res!;
                }

                // error
                if (cacheItem.hasOwnProperty('error')) {
                    throw cacheItem.error;
                }

                // waiting...
                throw cacheItem.promise || fetchData();
            },
            reload(force?: boolean) {
                if (force) {
                    // show loading status immediately
                    delete cacheItem.res;
                    delete cacheItem.error;
                    // load data
                    fetchData();
                    // force rerender component
                    rerender();
                } else {
                    // load data in background
                    fetchData();
                    // after data ready - render new data
                    cacheItem.promise!.then(rerender, rerender);
                }
            },
        },
    };

    function fetchData() {
        cacheItem.promise = fetchRequest<D, R, S>(url, data, { ...options, cache: cacheContext })
            .ready()
            .then((res) => {
                cacheItem.res = res;
                delete cacheItem.error;
            })
            .catch((error) => {
                cacheItem.error = error;

                delete cacheItem.res;
                // restore promise
            })
            .then(() => {
                const ttl = options?.ttl;

                if (ttl && ttl > 0) {
                    setTimeout(() => {
                        cache.delete(cacheKey);
                    }, ttl);
                }
            });

        return cacheItem.promise;
    }

    // initial data fetching
    if (!disabled) {
        fetchData();
    }

    cache.set(cacheKey, cacheItem);

    return cacheItem.pub;
}
