import React from 'react';

import { FetchError, fetchRequest, FetchRequestOptions } from 'shared/helpers/fetchRequest/fetchRequest';
import { useCacheRequestContext } from 'shared/hooks/useCacheRequestContext/useCacheRequestContext';

export interface UseFetchInfiniteState<T, R> {
    next?: T;
    data?: Optional<R>;
    error?: Optional<FetchError | Error>;
    isLoading: boolean;
}

export interface UseFetchInfiniteOptions<T, S, R> extends Omit<FetchRequestOptions<S, R>, 'parser'> {
    mergeData(prevData: Optional<R>, resData: S, state: UseFetchInfiniteState<T, R>): Optional<R>;
    mergeNext(prevNext: Optional<T>, resData: S, state: UseFetchInfiniteState<T, R>): Optional<T>;
}

export interface UseFetchInfiniteRes<T, R> extends Omit<UseFetchInfiniteState<T, R>, 'next'> {
    fetchNextPage(): void;
    updateData(updater: UseFetchDataUpdater<R>);
    isFullyLoaded: boolean;
}

export type UseFetchDataUpdater<R> = (data: Optional<R>) => Optional<R>;

export function mergeNextPageNumber<T extends { page_number?: number }, S extends { can_get_more_pages?: boolean }>(
    prevNext: Optional<T>,
    resData: S,
): Optional<T> {
    if (prevNext && resData.can_get_more_pages) {
        const nextPage = (prevNext.page_number ?? 1) + 1;

        return { ...prevNext, page_number: nextPage };
    }

    return undefined;
}

export function useFetchInfinite<T, R, S = R>(
    url: string,
    payload: T,
    options: UseFetchInfiniteOptions<T, S, R>,
): UseFetchInfiniteRes<T, R> {
    const { mergeNext, mergeData } = options;

    const cacheContext = useCacheRequestContext();

    const cancelRef = React.useRef<() => void>();
    const [state, setState] = React.useState<UseFetchInfiniteState<T, R>>({ isLoading: false, next: payload });

    const _fetch = React.useCallback(
        (payload: T, cleanup?: boolean) => {
            setState(cleanup ? () => ({ isLoading: true, next: payload }) : (state) => ({ ...state, isLoading: true }));

            const request = fetchRequest<T, S>(url, payload, { ...options, cache: cacheContext });

            request.ready().then(
                (data) =>
                    setState((state) => ({
                        next: mergeNext(state.next, data, state),
                        data: mergeData(cleanup ? undefined : state.data, data, state),
                        isLoading: false,
                    })),
                (error) => {
                    if (error instanceof FetchError && error.canceled) {
                        return;
                    }

                    setState((state) => ({ ...state, error, isLoading: false }));
                },
            );

            cancelRef.current = request.cancel;
        },
        [url, JSON.stringify(options)],
    );

    React.useEffect(() => {
        _fetch(payload, true);

        return () => {
            if (cancelRef.current) {
                cancelRef.current();
            }
        };
    }, [url, JSON.stringify(payload), JSON.stringify(options), _fetch]);

    const fetchNextPage = React.useCallback(() => {
        if (!state.isLoading && state.next) {
            _fetch(state.next);
        }
    }, [state, _fetch]);

    const updateData = React.useCallback((updater) => {
        setState((state) => ({
            ...state,
            data: updater(state.data),
        }));
    }, []);

    return React.useMemo(() => {
        return {
            data: state.data,
            error: state.error,
            isLoading: state.isLoading,
            isFullyLoaded: !state.isLoading && !state.next,
            fetchNextPage,
            updateData,
        };
    }, [state, fetchNextPage, updateData]);
}
