import { useCallback, useEffect, useMemo, useState } from 'react';

import { FetchError, fetchRequest, FetchRequestOptions } from 'shared/helpers/fetchRequest/fetchRequest';
import { useCacheRequestContext } from 'shared/hooks/useCacheRequestContext/useCacheRequestContext';

export interface UseFetchRequestOptions<S, R> extends FetchRequestOptions<S, R> {}

interface FetchState<R> {
    data?: Optional<R>;
    error?: Optional<FetchError | Error>;
    isLoading: boolean;
}

export function useFetchRequest<D, R, S = R>(
    url: string,
    data: D,
    options?: UseFetchRequestOptions<S, R>,
    disabled?: boolean,
) {
    const cacheContext = useCacheRequestContext();

    const [ver, setVer] = useState<number>(0);
    const [state, setState] = useState<FetchState<R>>({ isLoading: false });

    const reload = useCallback(() => {
        setVer((ver) => ver + 1);
    }, [setVer]);

    useEffect(() => {
        if (disabled) {
            return;
        }

        setState((state) => ({ ...state, isLoading: true }));

        const request = fetchRequest<D, R, S>(url, data, { ...options, cache: cacheContext });

        request.ready().then(
            (data) => setState(() => ({ data, isLoading: false })),
            (error) => {
                if (error instanceof FetchError && error.canceled) {
                    return;
                }

                setState(() => ({ error, isLoading: false }));
            },
        );

        return request.cancel;
    }, [url, JSON.stringify(data), JSON.stringify(options), disabled, ver]);

    return useMemo(() => {
        return {
            ...state,
            reload,
        };
    }, [state, reload]);
}
