import * as React from 'react';
import { flip } from '@floating-ui/react-dom';

import { MenuType } from 'shared/consts/MenuType';
import { useCheckedMenu } from 'shared/hooks/useCheckedMenu/useCheckedMenu';
import { useClickOutside } from 'shared/hooks/useClickOutside/useClickOutside';
import { useFloatingUI, UseFloatingUIOptions } from 'shared/hooks/useFloatingUI/useFloatingUI';
import { useKeyboardMenu } from 'shared/hooks/useKeyboardMenu/useKeyboardMenu';
import { useTimerRef } from 'shared/hooks/useTimerRef/useTimerRef';

const CLOSE_ANIMATION_TIME = 150;
const BLUR_TIME = 150;

export interface UseFloatingMenuOptions<T> {
    referenceRef: React.MutableRefObject<Nullable<HTMLElement>>;
    floatingRef: React.MutableRefObject<Nullable<HTMLElement>>;
    value?: string | string[];
    menuType?: MenuType;
    disabled?: boolean;
    options?: UseFloatingUIOptions;

    onReferenceBlur?(event: React.FocusEvent<T>): void;

    onReferenceClick?(event: React.MouseEvent<T>): void;

    onReferenceFocus?(event: React.FocusEvent<T>): void;

    onMenuChange?(checked: string | string[]): void;

    onMenuClick?(event: React.MouseEvent<HTMLLIElement>): void;

    onMenuClose?(): void;
}

// @todo: add test
export function useFloatingMenu<T>({
    referenceRef,
    floatingRef,
    value,
    menuType,
    disabled,
    options,
    onReferenceBlur,
    onReferenceClick,
    onReferenceFocus,
    onMenuChange,
    onMenuClick,
    onMenuClose,
}: UseFloatingMenuOptions<T>): {
    style: React.CSSProperties;
    isMenuClosing: boolean;
    isMenuVisible: boolean;
    checked: string | string[];
    onReferenceActionHandler(event: React.MouseEvent<T>): void;
    onReferenceBlurHandler(event: React.FocusEvent<T>): void;
    onReferenceFocusHandler(event: React.FocusEvent<T>): void;
    onMenuItemClickHandler(event: React.MouseEvent<HTMLLIElement>): void;
} {
    const [isMenuClosing, setMenuClosing] = React.useState<boolean>(false);
    const [isMenuVisible, setMenuVisible] = React.useState<boolean>(false);

    const timerRef = useTimerRef(0);
    const blurTimerRef = useTimerRef(0);
    const { style, update } = useFloatingUI(referenceRef, floatingRef, isMenuVisible, {
        ...options,
        middleware: [flip({ crossAxis: false, padding: 8 }), ...(options?.middleware || [])],
    });

    const onMenuCloseHandler = React.useCallback(() => {
        if (!isMenuClosing) {
            setMenuClosing(true);

            timerRef.current = setTimeout(() => {
                setMenuClosing(false);
                setMenuVisible(false);

                if (onMenuClose) {
                    onMenuClose();
                }
            }, CLOSE_ANIMATION_TIME) as unknown as number;
        }
    }, [onMenuClose, isMenuClosing]);

    const onOutsideClickHandler = React.useCallback(
        ({ target }: MouseEvent) => {
            if (!referenceRef.current?.contains(target as Node)) {
                onMenuCloseHandler();
            }
        },
        [referenceRef.current, onMenuCloseHandler],
    );

    const onReferenceBlurHandler = React.useCallback(
        (event: React.FocusEvent<T>) => {
            if (isMenuVisible && event.relatedTarget) {
                blurTimerRef.current = setTimeout(() => {
                    setMenuVisible(false);
                }, BLUR_TIME) as unknown as number;
            }

            if (onReferenceBlur) {
                onReferenceBlur(event);
            }
        },
        [onReferenceBlur, isMenuVisible, blurTimerRef],
    );

    const onReferenceActionHandler = React.useCallback(
        (event: React.MouseEvent<T>) => {
            if (disabled) {
                return;
            }

            if (isMenuVisible) {
                onMenuCloseHandler();
            } else {
                setMenuVisible(true);
                update();
            }

            if (onReferenceClick) {
                onReferenceClick(event);
            }
        },
        [disabled, onReferenceClick, isMenuVisible, onMenuCloseHandler],
    );

    const onReferenceFocusHandler = React.useCallback(
        (event: React.FocusEvent<T>) => {
            if (!isMenuVisible) {
                setMenuVisible(true);
                update();
            }

            if (onReferenceFocus) {
                onReferenceFocus(event);
            }
        },
        [onReferenceFocus, isMenuVisible],
    );

    const onMenuClickHandler = React.useCallback(
        (event: React.MouseEvent<HTMLLIElement>) => {
            if (isMenuVisible && menuType !== 'check') {
                onMenuCloseHandler();
            }

            if (onMenuClick) {
                onMenuClick(event);
            }
        },
        [menuType, onMenuClick, isMenuVisible, onMenuCloseHandler],
    );

    const { checked, onMenuItemClickHandler } = useCheckedMenu(value, menuType, onMenuClickHandler);

    useKeyboardMenu(onMenuCloseHandler, !isMenuVisible);
    useClickOutside(floatingRef, onOutsideClickHandler);

    React.useEffect(() => {
        if (onMenuChange) {
            onMenuChange(checked);
        }
    }, [JSON.stringify(checked)]);

    return {
        style,
        isMenuClosing,
        isMenuVisible,
        checked,
        onReferenceActionHandler,
        onReferenceFocusHandler,
        onReferenceBlurHandler,
        onMenuItemClickHandler,
    };
}
